<?php
/*
   This file is part of 'CInbox' (Common-Inbox)

   'CInbox' is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   'CInbox' is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with 'CInbox'.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once('include/CIFolder.php');
require_once('include/CIExec.php');
require_once('include/Helper.php');


/**
 * Abstract class containing functions/properties common to
 * Tasks that deal with copying/moving/renaming files from A to B.
 *
 *
 * @author Peter Bubestinger-Steindl (pb@av-rd.com)
 * @copyright
 *  Copyright 2016, Peter Bubestinger-Steindl
 *  (License: <a href="http://www.gnu.org/licenses/gpl.html">GNU General Public License (v3)</a>)
 *
 * @see
 *  - <a href="http://www.av-rd.com/">AV-RD website</a>
 *  - <a href="https://fsfe.org/about/basics/freesoftware.en.html">FSFE: What is Free Software?</a>
 *  - <a href="http://www.gnu.org/licenses/gpl.html">The GNU General Public License</a>
 */
abstract class TaskCopy extends CITask
{
    /* ========================================
     * CONSTANTS
     * ======================================= */

    // Task name/label:
    const TASK_LABEL = 'Copy files (abstract)';

    // Names of config settings used by a task must be defined here.
    const CONF_UPDATE_FOLDERS = 'UPDATE_FOLDERS';           // Defines how to treat existing/new folder handling
    const CONF_UPDATE_FILES = 'UPDATE_FILES';               // Similar to UPDATE_FOLDERS, but for files

    // Valid options for settings.
    // Folder/file target update options:
    const OPT_CREATE = 'create';
    const OPT_UPDATE = 'update';
    const OPT_CREATE_OR_UPDATE = 'create_or_update';

    // Misc:
    const MASK_TARGET_TEMP = 'temp_%s';



    /* ========================================
     * PROPERTIES
     * ======================================= */

    public static $OPT_UPDATE_FOLDERS  = array(
            self::OPT_CREATE,               // Create target folders. ERROR if target already exists.
            self::OPT_UPDATE,               // Target folder must already exist, but contents can be changed.
            );

    public static $OPT_UPDATE_FILES = array(
            self::OPT_CREATE,               // Only allow creating NEW files. ERROR if file already exists on target.
            self::OPT_UPDATE,               // Target files MUST already exist, but its contents will be overwritten by source.
            self::OPT_CREATE_OR_UPDATE,     // Target files CAN exist, but don't have to. Files are either created or overwritten.
            );

    protected $MASK_TARGET_TEMP = self::MASK_TARGET_TEMP;    // Use this mask to create temp target folders.

    # Variables that contain the settings from config file:
    protected $updateFolders;
    protected $updateFiles;



    /* ========================================
     * METHODS
     * ======================================= */

    function __construct(&$CIFolder)
    {
        parent::__construct($CIFolder, self::TASK_LABEL);

        // This is used for calling external copy-tool later:
        $this->exec = new CIExec();
    }



    /**
     * Prepare everything so it's ready for processing.
     * @return bool     success     True if init went fine, False if an error occurred.
     */
    public function init()
    {
        $l = $this->logger;

        if (!parent::init()) return false;

        // Name of logfile containing feedback from the external copy-command:
        $this->logFile = $this->getCmdLogfile();

        $this->targetFolderTemp = $this->CIFolder->getTargetFolder($this->MASK_TARGET_TEMP);
        $l->logDebug(sprintf(_("Target folder (temp) for '%s' resolves to: '%s'"), $this->CIFolder->getSubDir(), $this->targetFolderTemp));

        // Must return true on success:
        return true;
    }


    /**
     * Actions to be performed *after* run() finished successfully;
     */
    public function finalize()
    {
        if (!parent::finalize()) return false;

        // Remove logfile (if successful/no errors occurred):
        $this->removeCmdLogfile();

        // Must return true on success:
        return true;
    }


    /**
     * Load settings from config that are relevant for this task.
     */
    protected function loadSettings()
    {
        if (!parent::loadSettings()) return false;

        $l = $this->logger;
        $config = $this->config;

        $this->updateFolders = strtolower($config->get(self::CONF_UPDATE_FOLDERS));         // Normalize to lowercase.
        $l->logDebug(sprintf(_("Copy update mode (folders): %s"), $this->updateFolders));
        if (!$this->isValidUpdateFolders($this->updateFolders))
        {
            $l->logError(sprintf(
                        _("Invalid value set for '%s': %s"),
                        self::CONF_UPDATE_FOLDERS,
                        $this->updateFolders));
            $this->setStatusConfigError();
            return false;
        }

        $this->updateFiles = strtolower($config->get(self::CONF_UPDATE_FILES));             // Normalize to lowercase.
        $l->logDebug(sprintf(_("Copy update mode (files): %s"), $this->updateFiles));
        if (!$this->isValidUpdateFiles($this->updateFiles))
        {
            $l->logError(sprintf(_("Invalid value set for '%s': %s"),
                        self::CONF_UPDATE_FILES,
                        $this->updateFiles));
            $this->setStatusConfigError();
            return false;
        }

        // Must return true on success:
        return true;
    }



    // --------------------------------------------
    // Task-specific methods
    // --------------------------------------------

    /**
     * Checks if the given update mode is a valid option.
     * Returns 'true' if it is valid - 'false' if not.
     *
     * The check is case-sensitive, so you might want to normalize case before calling this.
     */
    protected function isValidUpdateFolders($updateMode)
    {
        return in_array($updateMode, static::$OPT_UPDATE_FOLDERS);
    }


    /**
     * Same as "isValidUpdateFolders()" but for file mode.
     */
    protected function isValidUpdateFiles($updateMode)
    {
        return in_array($updateMode, static::$OPT_UPDATE_FILES);
    }


    /**
     * Get filename of logfile to use for external command execution.
     * NOTE: Assumes $this->tempFolder to be set and initialized.
     *
     * @See: self::checkTempFolder()
     */
    protected function getCmdLogfile()
    {
        // Check if we have a proper temp folder:
        if (!$this->checkTempFolder()) return false;

        $logFile = Helper::resolveString(
                $this->tempFolder . DIRECTORY_SEPARATOR . $this->name . __DATETIME__ . '.log',
                array(__DATETIME__ => date('Ymd_His'))
                );

        return $logFile;
    }


    /**
     * Removes/deletes the logfile used for external command execution.
     * @See self::getCmdLogfile()
     */
    protected function removeCmdLogfile()
    {
        $l = $this->logger;
        $logfile = $this->logFile;

        if (!file_exists($logfile))
        {
            // Logfile isn't there, so remove can report "success" ;)
            $l->logDebug(sprintf(_("Logfile '%s' removed, since it didn't exist in the first place ;)"), $logfile));
            return true;
        }

        if (!is_writable($this->logFile))
        {
            $l->logError(sprintf(_("Logfile '%s' cannot be removed, because it is not writable. Check access rights?"), $logfile));
            // Mark this "problem but continue", since it is non-critical to this or following tasks:
            $this->setStatusPBC();
            return false;
        }

        if (!unlink($logfile))
        {
            $l->logError(sprintf(_("Logfile '%s' could be removed. Unknown reason."), $logfile));
            // Mark this "problem but continue", since it is non-critical to this or following tasks:
            $this->setStatusPBC();
            return false;
        }

        $l->logDebug(sprintf(_("Logfile '%s' was removed."), $logfile));
        return true;
    }


    /**
     * Creates a folder.
     * Performs checks before doing so and writes to log.
     * Returns 'true' if folder was created, 'false' if not.
     */
    protected function createFolder($folderName)
    {
        $l = $this->logger;

        if (empty($folderName))
        {
            // This is not an exception, so that task execution *may* still proceed in case of this error:
            // TODO: Throw custom exception. Seems cleaner.
            $l->logError(_("createFolder: Empty folder name. This should not happen."));
            $this->setStatusError();
            return false;
        }

        $l->logMsg(sprintf(_("Creating folder '%s'..."), $folderName));
        if (is_dir($folderName)) return true;

        # TODO: Create parent folders, too. CAUTION: Inhert permissions from existing parent folders.
        if (!mkdir($folderName))
        {
            $l->logErrorPhp(sprintf(_("Failed to create folder '%s'."), $folderName));
            return false;
        }

        return true;
    }


    /**
     * Renames/moves a file or folder from $source to $target.
     *
     * It uses the PHP built-in function "rename", but different checks:
     * $overwrite:
     *   If false: If $target already exists, exception will be thrown.
     *   If true:  If $target already exists it will be overwritten by $source.
     */
    protected function move($source, $target, $overwrite=false)
    {
        $l = $this->logger;

        if (empty($source)) throw new Exception(_("move: Empty source given. This should not happen."));
        if (empty($target)) throw new Exception(_("move: Empty target given. This should not happen."));
        if (!file_exists($source)) throw new Exception(sprintf(_("Source does not exist: '%s'."), $source));

        if (file_exists($target))
        {
            if (!$overwrite) throw new Exception(sprintf(_("Target already exists: '%s'."), $target));
            // Overwrite = 1. delete old file / 2. move new file into position.
            if (!unlink($target))
            {
                $l->logErrorPhp(sprintf(_("Unable to overwrite '%s'."), $target));
                $this->setStatusPBCT();
                return false;
            }
        }

        $l->logMsg(sprintf(_("Moving '%s' to '%s'..."), $source, $target));

        if (!rename($source, $target))
        {
            $l->logErrorPhp(sprintf(_("Failed to rename '%s' to '%s'."), $source, $target));
            return false;
        }

        return true;
    }



}

?>
