<?php
/*
   This file is part of 'CInbox' (Common-Inbox)

   'CInbox' is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   'CInbox' is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with 'CInbox'.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once('include/CIFolder.php');
require_once('include/CIExec.php');
require_once('include/Helper.php');


/**
 * This task copies the files from source folder to target folder.
 *
 * The actual file transfer process is done by 'rsync' to ensure reliable data transfer.
 * @See TaskCopyRsync
 *
 *
 * @author Peter Bubestinger-Steindl (pb@av-rd.com)
 * @copyright
 *  Copyright 2016, Peter Bubestinger-Steindl
 *  (License: <a href="http://www.gnu.org/licenses/gpl.html">GNU General Public License (v3)</a>)
 *
 * @see
 *  - <a href="http://www.av-rd.com/">AV-RD website</a>
 *  - <a href="https://fsfe.org/about/basics/freesoftware.en.html">FSFE: What is Free Software?</a>
 *  - <a href="http://www.gnu.org/licenses/gpl.html">The GNU General Public License</a>
 */
class TaskCopyToTarget extends TaskCopyRsync
{
    /* ========================================
     * CONSTANTS
     * ======================================= */

    // Task name/label:
    const TASK_LABEL = 'Copy to target';



    /* ========================================
     * PROPERTIES
     * ======================================= */



    /* ========================================
     * METHODS
     * ======================================= */

    function __construct(&$CIFolder)
    {
        parent::__construct($CIFolder, self::TASK_LABEL);
    }



    /**
     * Prepare everything so it's ready for processing.
     * @return bool     success     True if init went fine, False if an error occurred.
     */
    public function init()
    {
        $l = $this->logger;

        if (!parent::init()) return false;

        // Must return true on success:
        return true;
    }


    /**
     * Perform the actual steps of this task.
     */
    public function run()
    {
        if (!parent::run()) return false;

        // Copy files of this folder to targetFolderTemp:
        if (!$this->copyFolder($this->sourceFolder, $this->targetFolder, $this->targetFolderTemp)) return false;

        // Must return true on success:
        $this->setStatusDone();
        return true;
    }



    // --------------------------------------------
    // Task-specific methods
    // --------------------------------------------

    protected function copyFolder($sourceFolder, $targetFolder, $targetFolderTemp)
    {
        $l = $this->logger;
        $errors = 0;
        $entryCount = 0;
        $copyCount = 0;

        // Verify if desired target folder is okay:
        if (!$this->checkTargetFolderCondition($targetFolder, $this->updateFolders)) return false;
        if (!$this->createFolder($targetFolderTemp))
        {
            $this->setStatusPBCT();
            return false;
        }

        $list = Helper::getFolderListing2($sourceFolder);
        ksort($list); // sort filenames alphabetically

        // Iterate through listing of entries in current folder:
        foreach ($list as $key=>$entry)
        {
            // Only iterate through files. Each task only handles the current folder (CIFolder).
            if (is_dir($key)) continue;

            $entryCount++;
            $sourceFile = $key;
            $targetFile = Helper::getTargetFilename($sourceFile, $targetFolder);

            $targetFileTemp = Helper::getTargetFilename($sourceFile, $targetFolderTemp);

            if (!$this->checkTargetFileCondition($targetFile, $this->updateFiles))
            {
                $errors++;
                continue;
            }

            // Skip file if it matches 'CONF_COPY_EXCLUDE' patterns:
            $exclude = $this->exclude($sourceFile, $this->copyExclude);
            if ($exclude !== false)
            {
                $l->logMsg(sprintf(_("Excluding file '%s'. Matches pattern '%s'."), $sourceFile, $exclude));
                continue;
            }

            $l->logMsg(sprintf(_("Copying '%s' to '%s'..."), $sourceFile, $targetFileTemp));
            $exitCode = $this->copyFile($sourceFile, $targetFileTemp);
            ($exitCode == CIExec::EC_OK) ? $copyCount++ : $errors++;
        }

        if ($entryCount == 0)
        {
            $l->logMsg(sprintf(_("No files to copy in '%s'. Fine."), $sourceFolder));
            return true;
        }

        $l->logMsg(sprintf(
                    _("%d/%d file(s) copied from '%s' to '%s'."),
                    $copyCount,
                    $entryCount,
                    $sourceFolder,
                    $targetFolder));

        if ($errors > 0)
        {
            $l->logError(sprintf(
                        _("%d errors encountered trying to copy '%s' to '%s'."),
                        $errors,
                        $sourceFolder,
                        $targetFolder));
            $this->setStatusPBCT();

            // Remove empty subfolders on targetFolderTemp (recursive rmdir?).
            $l->logInfo(sprintf(_("Removing empty temp-folders in '%s'..."), $targetFolderTemp));
            Helper::removeEmptySubfolders($targetFolderTemp);
            return false;
        }

        return true;
    }


    /**
     * Checks the current situation of target folder, according
     * to the setting given in $updateFolders.
     *
     * NOTE: Condition mismatch is treated as "STATUS_PBC".
     * This allows an operator to clear all reported warnings/errors before resetting
     * the Item, as the task will try to process all subfolders and report *all*
     * possibly mismatching entries.
     *
     * $updateFolders must contain a valid option from self::$OPT_UPDATE_FOLDERS.
     * @see self::isValidUpdateFolders()
     */
    protected function checkTargetFolderCondition($targetFolder, $updateFolders)
    {
        $l = $this->logger;

        if (empty($targetFolder))
        {
            $l->logError(_('checkTargetFolderCondition: Empty target folder given.'));
            // TODO: Actually, this should not happen - so it should be an exception, rather than task-error?
            $this->setStatusError();
            return false;
        }

        if (empty($updateFolders))
        {
            $l->logError(_('checkTargetFolderCondition: $updateFolders empty.'));
            // TODO: Actually, this should not happen - so it should be an exception, rather than task-error?
            $this->setStatusConfigError();
            return false;
        }

        // The target may *never* be a file. Regardless of $updateFolders setting:
        if (file_exists($targetFolder))
        {
            $l->logInfo(sprintf(_("Target folder '%s' already exists."), $targetFolder));

            if (!is_dir($targetFolder))
            {
                $l->logError(sprintf(_("Target folder '%s' is a file, but must be a folder. Not good."), $targetFolder));
                $this->setStatusError();
                return false;
            }

            if (!is_writable($targetFolder))
            {
                $l->logError(sprintf(_("Target folder '%s' is not writable. Check access rights?"), $targetFolder));
                $this->setStatusError();
                return false;
            }
        }

        switch ($updateFolders)
        {
            // ----------------------------------
            case self::OPT_CREATE:
                if (file_exists($targetFolder))
                {
                    $l->logError(sprintf(
                                _("Folder '%s' already exists. This violates config condition '%s = %s'."),
                                $targetFolder,
                                self::CONF_UPDATE_FOLDERS,
                                $updateFolders));
                    $this->setStatusPBCT();
                    return false;
                }
                return true;
                break;

                // ----------------------------------
            case self::OPT_UPDATE:
                if (!file_exists($targetFolder))
                {
                    $l->logError(sprintf(
                                _("Target folder '%s' does not exist, but has to. This violates config condition '%s = %s'."),
                                $targetFolder,
                                self::CONF_UPDATE_FOLDERS,
                                $updateFolders));
                    $this->setStatusPBCT();
                    return false;
                }
                return true;
                break;

                // ----------------------------------
            default:
                // If option is valid should have been checked in "loadSettings()".
                throw new Exception(sprintf(
                            _("Invalid option for '%s': %s.\nThis should not have happened."),
                            self::CONF_UPDATE_FOLDERS,
                            $updateFolders
                            ));
        }

        $l->logError(_('checkTargetFolderCondition: No matching rule. This is odd and should not have happened.'));
        $this->setStatusError();
        return false;
    }


    /**
     * Similar to checkTargetFolderCondition(), but for target files instead of folders.
     * Scope is only one folder (current task's $CIFolder).
     *
     * @see self::OPT_UPDATE_FILES
     */
    protected function checkTargetFileCondition($targetFile, $updateFiles)
    {
        $l = $this->logger;

        // The target may *never* be a folder. Regardless of $updateFiles setting:
        if (file_exists($targetFile))
        {
            $l->logInfo(sprintf(_("Target file '%s' already exists."), $targetFile));

            if (is_dir($targetFile))
            {
                $l->logError(sprintf(_("Target '%s' is a folder, but should be a file. Not good."), $targetFile));
                $this->setStatusError();
                return false;
            }

            if (!is_writable($targetFile))
            {
                $l->logError(sprintf(_("Target '%s' is not writable. Check access rights?"), $targetFile));
                $this->setStatusError();
                return false;
            }
        }

        switch ($updateFiles)
        {
            case self::OPT_CREATE:
                if (file_exists($targetFile))
                {
                    $l->logError(sprintf(
                                _("File '%s' already exists. This violates config condition '%s = %s'."),
                                $targetFile,
                                self::CONF_UPDATE_FILES,
                                $updateFiles));
                    $this->setStatusPBCT();
                    return false;
                }
                return true;
                break;

            case self::OPT_UPDATE:
                if (!file_exists($targetFile))
                {
                    $l->logError(sprintf(
                                _("File '%s' does NOT exist, but has to. This violates config condition '%s = %s'."),
                                $targetFile,
                                self::CONF_UPDATE_FILES,
                                $updateFiles));
                    $this->setStatusPBCT();
                    return false;
                }
                return true;
                break;

            case self::OPT_CREATE_OR_UPDATE:
                // Target files CAN exist, but don't have to. Files are either created or overwritten.
                return true;
                break;

            default:
                // If option is valid should have been checked in "loadSettings()".
                throw new Exception(sprintf(
                            _("Invalid option for '%s': %s.\nThis should not have happened."),
                            self::CONF_UPDATE_FILES,
                            $updateFiles
                            ));
        }

        $l->logError(_('checkTargetFileCondition: No matching rule. This is odd and should not have happened.'));
        $this->setStatusError();
        return false;
    }



}

?>
