/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include "bmio_driver.h"
#include "bmio_subdriver.h"
#include "bm_pci.h"
#include "bm_version.h"
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <linux/sched.h>

#ifdef __i386__
	// 32-bit systems may not have a 64-bit cmpxchg function, so limit the
	// supported number of ids to 32.
	typedef uint32_t device_mask_id_t;
#else
	typedef uint64_t device_mask_id_t;
#endif

extern struct file_operations bmio_fops;

static device_mask_id_t blackmagic_device_ids = 0;

static LIST_HEAD(bmio_drivers);
static LIST_HEAD(bmio_subdrivers);
static DEFINE_MUTEX(bmio_drivers_lock);

static int bmio_alloc_id(void)
{
	int id;
	device_mask_id_t mask;
	device_mask_id_t old_id_map;

	for (id = 0; id < sizeof(device_mask_id_t) * 8; /* nothing */)
	{
		mask = (1UL << id);

		old_id_map = blackmagic_device_ids;
		if (!(old_id_map & mask))
		{
			if (cmpxchg(&blackmagic_device_ids, old_id_map, old_id_map | mask) == old_id_map)
				return id;
		}
		else
		{
			++id;
		}
	}

	return -1;
}

static void bmio_release_id(int id)
{
	device_mask_id_t old_id_map;

	do
	{
		old_id_map = blackmagic_device_ids;

		if (cmpxchg(&blackmagic_device_ids, old_id_map, old_id_map & ~(1UL << id)) == old_id_map)
			break;
	}
	while (true);
}

bmio_driver_t* bmio_driver_register(BlackmagicIODriverLinuxPtr driver, bmio_device_t* dev)
{
	bmio_driver_t* drv = NULL;

	drv = kzalloc(sizeof(bmio_driver_t), GFP_KERNEL);
	if (!drv)
		return NULL;

	drv->id = -1;
	INIT_LIST_HEAD(&drv->list);
	drv->activated = false;
	drv->driver = driver;
	drv->dev = bmio_device_get(dev);
	init_waitqueue_head(&drv->wait);

	drv->id = bmio_alloc_id();
	if (drv->id < 0)
		goto bail;

	drv->name[DRIVER_NAME_MAX_SIZE - 1] = '\0';
	snprintf(drv->name, DRIVER_NAME_MAX_SIZE - 1, "blackmagic!io%u", drv->id);

	drv->mdev = (struct miscdevice) {
		.minor = MISC_DYNAMIC_MINOR,
		.name = drv->name,
		.fops = &bmio_fops,
#if KERNEL_VERSION_OR_LATER(2, 6, 20)
		.parent = &dev->pci->pdev->dev,
#endif
#if KERNEL_VERSION_OR_LATER(2, 6, 32)
		.mode = 0666,
#endif
	};

	mutex_lock(&bmio_drivers_lock);
		list_add(&drv->list, &bmio_drivers);
	mutex_unlock(&bmio_drivers_lock);

	if (misc_register(&drv->mdev) != 0)
		goto bail;

	return drv;

bail:
	if (drv->id >= 0)
		bmio_release_id(drv->id);
	if (!list_empty(&drv->list))
	{
		mutex_lock(&bmio_drivers_lock);
			list_del(&drv->list);
		mutex_unlock(&bmio_drivers_lock);
	}
	bmio_device_put(drv->dev);
	kfree(drv);
	return NULL;
}

bool bmio_driver_started(bmio_driver_t* drv)
{
	bm_pci_device_t* pci = bmio_driver_pci(drv);
	if (!pci)
		return false;

	printk(KERN_NOTICE "BlackmagicIO: %s as %s [%04x:%02x:%02x.%d]\n",
		bmio_driver_name(drv),
		drv->mdev.name,
		pci_domain_nr(pci->pdev->bus),
		pci->pdev->bus->number,
		PCI_SLOT(pci->pdev->devfn),
		PCI_FUNC(pci->pdev->devfn));

	bmio_subdriver_probe(drv);

	return true;
}

void bmio_driver_stopping(bmio_driver_t* drv)
{
	bm_pci_device_t* pci = bmio_driver_pci(drv);

	printk(KERN_NOTICE "BlackmagicIO: Removing %s [%04x:%02x:%02x.%d]\n",
		drv->mdev.name,
		pci_domain_nr(pci->pdev->bus),
		pci->pdev->bus->number,
		PCI_SLOT(pci->pdev->devfn),
		PCI_FUNC(pci->pdev->devfn));
}

void bmio_driver_unregister(bmio_driver_t* drv)
{
	bmio_subdriver_remove(drv);

	misc_deregister(&drv->mdev);

	mutex_lock(&bmio_drivers_lock);
		list_del(&drv->list);
	mutex_unlock(&bmio_drivers_lock);

	bmio_release_id(drv->id);

	bmio_device_put(drv->dev);

	kfree(drv);
}

void bmio_driver_activate(bmio_driver_t* drv)
{
	drv->activated = true;
	wake_up_all(&drv->wait);
}

bool bmio_driver_activated(bmio_driver_t* drv)
{
	return wait_event_interruptible(drv->wait, drv->activated) == 0;
}

bmio_driver_t* bmio_driver_find_by_inode(struct inode* inode)
{
	bmio_driver_t* drv;
	int minor = iminor(inode);

	mutex_lock(&bmio_drivers_lock);
		list_for_each_entry(drv, &bmio_drivers, list)
		{
			if (drv->mdev.minor == minor)
				goto done;
		}
		drv = NULL;

done:
	mutex_unlock(&bmio_drivers_lock);
	return drv;
}

int bmio_driver_id(bmio_driver_t* drv)
{
	return drv->id;
}
EXPORT_SYMBOL(bmio_driver_id);

const char* bmio_driver_device_name(bmio_driver_t* drv)
{
	return drv->name;
}
EXPORT_SYMBOL(bmio_driver_device_name);

EXPORT_SYMBOL(bmio_driver_name);

struct device* bmio_driver_device(bmio_driver_t* drv)
{
	bm_pci_device_t* pci = bmio_driver_pci(drv);
	if (!pci)
		return NULL;

	return &pci->pdev->dev;
}
EXPORT_SYMBOL(bmio_driver_device);

bm_pci_device_t* bmio_driver_pci(bmio_driver_t* drv)
{
	return drv->dev->pci;
}

//
// Subdriver handing
//

int bmio_driver_register_subdriver(bmio_subdriver_t* subdriver)
{
	bmio_driver_t* drv;

	mutex_lock(&bmio_drivers_lock);
		list_add(&subdriver->list, &bmio_subdrivers);

		list_for_each_entry(drv, &bmio_drivers, list)
		{
			if (subdriver->probe(drv) < 0)
				printk(KERN_WARNING "BlackmagicIO: Failed to initialise %s driver for %s\n", subdriver->name, drv->mdev.name);
			else
				printk(KERN_INFO "BlackmagicIO: Initialised %s driver for %s\n", subdriver->name, drv->mdev.name);
		}
	mutex_unlock(&bmio_drivers_lock);

	return 0;
}
EXPORT_SYMBOL(bmio_driver_register_subdriver);

void bmio_driver_unregister_subdriver(bmio_subdriver_t* subdriver)
{
	bmio_driver_t* drv;

	mutex_lock(&bmio_drivers_lock);
		list_for_each_entry(drv, &bmio_drivers, list)
		{
			subdriver->remove(drv);
		}

		list_del(&subdriver->list);
	mutex_unlock(&bmio_drivers_lock);
}
EXPORT_SYMBOL(bmio_driver_unregister_subdriver);

void bmio_subdriver_probe(bmio_driver_t* drv)
{
	bmio_subdriver_t* subdriver;

	mutex_lock(&bmio_drivers_lock);
		list_for_each_entry(subdriver, &bmio_subdrivers, list)
		{
			if (subdriver->probe(drv) < 0)
				printk(KERN_WARNING "BlackmagicIO: Failed to initialise subdriver %s for %s\n", subdriver->name, drv->mdev.name);
			else
				printk(KERN_INFO "BlackmagicIO: Initialised %s driver for %s\n", subdriver->name, drv->mdev.name);
		}
	mutex_unlock(&bmio_drivers_lock);
}

void bmio_subdriver_remove(bmio_driver_t* drv)
{
	bmio_subdriver_t* subdriver;

	mutex_lock(&bmio_drivers_lock);
		list_for_each_entry(subdriver, &bmio_subdrivers, list)
		{
			subdriver->remove(drv);
		}
	mutex_unlock(&bmio_drivers_lock);
}

