/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include "bmio_device.h"
#include <linux/pci.h>

bmio_device_t* bmio_device_alloc(struct pci_dev* pdev)
{
	bmio_device_t* dev = kzalloc(sizeof(bmio_device_t), GFP_KERNEL);
	if (!dev)
		return NULL;

	atomic_set(&dev->ref, 1);
	mutex_init(&dev->lock);

	dev->pci = bm_pci_alloc(pdev);
	if (!dev->pci)
		goto bail;

	pci_set_drvdata(pdev, dev);

	if (bmio_device_init(dev) != 0)
		goto bail;

	return dev;

bail:
	pci_set_drvdata(pdev, NULL);
	if (dev->pci)
		bm_pci_put(dev->pci);
	return NULL;
}

void bmio_device_remove(bmio_device_t* dev)
{
	mutex_lock(&dev->lock);
		dev->removed = true;
	mutex_unlock(&dev->lock);

	bmio_device_deinit(dev);
	bmio_device_put(dev);
}

bmio_device_t* bmio_device_get(bmio_device_t* dev)
{
	atomic_inc(&dev->ref);
	return dev;
}

void bmio_device_put(bmio_device_t* dev)
{
	if (atomic_sub_and_test(1, &dev->ref))
	{
		bm_pci_put(dev->pci);
		kfree(dev);
	}
}

void bmio_device_pci_name(bmio_device_t* dev, char* buf, size_t len)
{
	buf[len - 1] = '\0';
	snprintf(buf, len - 1, "PCI[%04x:%02x:%02x.%d]",
		pci_domain_nr(dev->pci->pdev->bus),
		dev->pci->pdev->bus->number,
		PCI_SLOT(dev->pci->pdev->devfn),
		PCI_FUNC(dev->pci->pdev->devfn)
	);
}
