/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/version.h>
#include "bmio_device.h"
#include "bmio_serial.h"
#include "bmio_subdriver.h"
#include "bm_util.h"
#include "blackmagic.h"

extern struct pci_device_id bmio_pci_ids[];

static int bmio_probe(struct pci_dev *pdev, const struct pci_device_id *id)
{
	bmio_device_t* dev = bmio_device_alloc(pdev);
	if (!dev)
		return -ENOMEM;

	return 0;
}

static void bmio_remove(struct pci_dev *pdev)
{
	bmio_device_t* dev = (bmio_device_t*)pci_get_drvdata(pdev);
	if (dev)
		bmio_device_remove(dev);
}

static int bmio_suspend(struct pci_dev *pdev, pm_message_t state)
{
	bmio_device_t* dev = (bmio_device_t*)pci_get_drvdata(pdev);
	if (dev)
		return bmio_device_suspend(dev);
	return 0;
}

static int bmio_resume(struct pci_dev *pdev)
{
	bmio_device_t* dev = (bmio_device_t*)pci_get_drvdata(pdev);
	if (dev)
		return bmio_device_resume(dev);
	return 0;
}

static void bmio_shutdown(struct pci_dev *pdev)
{
	bmio_suspend(pdev, PMSG_SUSPEND);
}

static struct pci_driver pci_driver =
{
	.name = "blackmagic-io",
	.id_table = bmio_pci_ids,
	.probe = bmio_probe,
	.remove = bmio_remove,
	.shutdown = bmio_shutdown,
	.suspend = bmio_suspend,
	.resume = bmio_resume,
};

static int __init bmio_init(void)
{
	int ret;

	if (!blackmagic_init())
		return -ENOMEM;

	printk(KERN_NOTICE "BlackmagicIO: Driver version 12.9a3 loading\n");

	ret = bmio_serial_driver_init();
	if (ret)
		return ret;

	ret = pci_register_driver(&pci_driver);
	if (ret)
	{
		bmio_serial_driver_exit();
		return ret;
	}

	return 0;
}

static void __exit bmio_exit(void)
{
	pci_unregister_driver(&pci_driver);
	bmio_serial_driver_exit();
	blackmagic_exit();
}

MODULE_AUTHOR("Blackmagic Design Inc. <developer@blackmagicdesign.com>");
MODULE_DESCRIPTION("Blackmagic Design BlackmagicIO driver");
MODULE_VERSION("12.9a3");
MODULE_LICENSE("Proprietary");

module_init(bmio_init);
module_exit(bmio_exit);
