/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <asm/io.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include "bm_version.h"
#if KERNEL_VERSION_OR_LATER(4, 11, 0)
	#include <linux/sched/signal.h>
#endif
#include "bmio_client.h"
#include "bm_util.h"

bmio_client_t* bmio_client_alloc(bmio_driver_t* drv)
{
	bmio_client_t* cli = kzalloc(sizeof(bmio_client_t), GFP_KERNEL);
	if (!cli)
		return NULL;

	atomic_set(&cli->ref, 1);
	INIT_LIST_HEAD(&cli->notify.head);
	init_waitqueue_head(&cli->notify.wait);
	cli->notify.count = 0;

	cli->notify.lock = bm_spin_alloc();
	if (!cli->notify.lock)
		goto bail;

	if (bmio_client_init(cli, drv) != 0)
		goto bail;

	return cli;

bail:
	if (cli)
		bmio_client_put(cli);
	return NULL;
}

bmio_client_t* bmio_client_get(bmio_client_t* cli)
{
	atomic_inc(&cli->ref);
	return cli;
}

void bmio_client_put(bmio_client_t* cli)
{
	if (atomic_sub_and_test(1, &cli->ref))
	{
		unsigned long flags;
		bm_spin_lock_irqsave(cli->notify.lock, &flags);
			bmio_notify_clear(&cli->notify);
		bm_spin_unlock_irqrestore(cli->notify.lock, flags);

		bm_spin_free(cli->notify.lock);

		kfree(cli);
	}
}

static int bmio_open(struct inode *inode, struct file *filp)
{
	bmio_client_t* cli;
	bmio_driver_t* drv;
	int err = 0;

#if KERNEL_VERSION_OR_LATER(3, 9, 0)
	drv = bmio_driver_find_by_inode(file_inode(filp));
#elif KERNEL_VERSION_OR_LATER(2, 6, 36)
	drv = bmio_driver_find_by_inode(filp->f_dentry->d_inode);
#else
	drv = bmio_driver_find_by_inode(inode);
#endif
	if (!drv)
		return -ENODEV;

	mutex_lock(&drv->dev->lock);

		if (drv->dev->removed)
		{
			err = -ENODEV;
			goto bail;
		}

		if (!bmio_driver_activated(drv))
		{
			err = -EACCES;
			goto bail;
		}

		cli = bmio_client_alloc(drv);
		if (!cli)
		{
			err = -ENOMEM;
			goto bail;
		}

		filp->private_data = cli;

bail:
	mutex_unlock(&drv->dev->lock);
	return err;
}

static int bmio_release(struct inode *inode, struct file *filp)
{
	bmio_client_t* cli = filp->private_data;
	if (!cli)
		return -ENODEV;

	bmio_client_deinit(cli);

	bmio_client_put(cli);

	return 0;
}

static long bmio_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	bmio_client_t* cli = filp->private_data;
	if (!cli)
		return -ENODEV;

	return bmio_client_ioctl(cli, cmd, arg);
}

static unsigned int bmio_poll(struct file *filp, poll_table *wait)
{
	unsigned int mask = 0;
	unsigned long flags;

	bmio_client_t* cli = filp->private_data;
	if (!cli)
		return 0;

	poll_wait(filp, &cli->notify.wait, wait);

	bm_spin_lock_irqsave(cli->notify.lock, &flags);
		if (!list_empty(&cli->notify.head))
			mask = POLLIN | POLLRDNORM;
	bm_spin_unlock_irqrestore(cli->notify.lock, flags);

	return mask;
}

static ssize_t bmio_read(struct file* filp, char __user* buffer, size_t count, loff_t* ppos)
{
	ssize_t written = 0;
	unsigned long flags;

	bmio_client_t* cli = filp->private_data;
	if (!cli)
		return -ENODEV;

	bm_spin_lock_irqsave(cli->notify.lock, &flags);

	if (!(filp->f_flags & O_NONBLOCK) && list_empty(&cli->notify.head))
	{
		DEFINE_WAIT(wait);

		while (true)
		{
			prepare_to_wait(&cli->notify.wait, &wait, TASK_INTERRUPTIBLE);

			if (!list_empty(&cli->notify.head))
				break;

			if (!signal_pending(current))
			{
				bm_spin_unlock_irq(cli->notify.lock);
					schedule();
				bm_spin_lock_irq(cli->notify.lock);
				continue;
			}

			written = -ERESTARTSYS;
			break;
		}

		finish_wait(&cli->notify.wait, &wait);

		if (written)
			goto done;
	}
	else if ((filp->f_flags & O_NONBLOCK) && !list_empty(&cli->notify.head))
	{
		written = -EAGAIN;
		goto done;
	}

	written = bmio_notify_read(&cli->notify, buffer, count);
	if (written > 0)
		*ppos += written;

done:
	bm_spin_unlock_irqrestore(cli->notify.lock, flags);

	return written;
}

static void bmio_mmap_open(struct vm_area_struct *vma)
{
	bmio_client_mmap_t* mmap = (bmio_client_mmap_t*)vma->vm_private_data;
	atomic_inc(&mmap->ref);
}

static void bmio_mmap_close(struct vm_area_struct *vma)
{
	bmio_client_mmap_t* mmap = (bmio_client_mmap_t*)vma->vm_private_data;
	if (atomic_sub_and_test(1, &mmap->ref))
	{
		bmio_client_munmap(mmap->cli, mmap);
		bmio_client_put(mmap->cli);
		kfree(mmap);
	}
}

static struct vm_operations_struct bmio_client_vm_ops =
{
	.open = bmio_mmap_open,
	.close = bmio_mmap_close,
};

static int bmio_mmap(struct file *filp, struct vm_area_struct *vma)
{
	int r;
	bmio_client_mmap_t* mmap;

	bmio_client_t* cli = filp->private_data;
	if (!cli)
		return -ENODEV;

	mmap = kzalloc(sizeof(bmio_client_mmap_t), GFP_KERNEL);
	if (!mmap)
		return -ENOMEM;

	*mmap = (bmio_client_mmap_t)
	{
		.cli = bmio_client_get(cli),
		.type = vma->vm_pgoff,
		.size = vma->vm_end - vma->vm_start,
	};

	atomic_set(&mmap->ref, 1);

	r = bmio_client_mmap(cli, mmap);
	if (r != 0)
	{
		bmio_client_put(mmap->cli);
		kfree(mmap);
		return r;
	}

	r = remap_pfn_range(vma, vma->vm_start, __pa(mmap->buffer) >> PAGE_SHIFT, mmap->size, vma->vm_page_prot);
	if (r != 0)
	{
		bmio_client_munmap(cli, mmap);
		kfree(mmap);
		return r;
	}

	vma->vm_private_data = mmap;
	vma->vm_ops = &bmio_client_vm_ops;

	return r;
}

struct file_operations bmio_fops =
{
	.owner = THIS_MODULE,
	.open = bmio_open,
	.release = bmio_release,
	.unlocked_ioctl = bmio_ioctl,
	.compat_ioctl = bmio_ioctl,
	.poll = bmio_poll,
	.read = bmio_read,
	.mmap = bmio_mmap,
};

bool bmio_client_notify(bmio_client_t* cli, uint32_t type, uint32_t param1, uint32_t param2, uint32_t param3, uint32_t param4)
{
	if (!bmio_notify_add(&cli->notify, type, param1, param2, param3, param4))
		return false;

	wake_up_interruptible(&cli->notify.wait);

	return true;
}

size_t bmio_client_notify_queue_size(bmio_client_t* cli)
{
	unsigned long flags;
	size_t count;

	bm_spin_lock_irqsave(cli->notify.lock, &flags);
		count = cli->notify.count;
	bm_spin_unlock_irqrestore(cli->notify.lock, flags);
	return count;
}
