/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#ifndef BM_UTIL_H
#define BM_UTIL_H

#include "bm_types.h"


void bm_util_init(void);

// Basic memory management
void *bm_kmalloc(size_t size);
void *bm_kzalloc(size_t size);
void bm_kfree(void* mem);

void* bm_vmalloc(size_t size);
void bm_vfree(void* mem);

#define BM_ALLOC_CONTIGUOUS		0x80000000
#define BM_ALLOC_ALIGNMENT_MASK	0x0000FFFF
void* bm_alloc_aligned(size_t size, uint32_t flags);
void bm_free_aligned(void* mem);
uint32_t bm_aligned_flags(void* mem);

// Logging
extern const char* BM_ERR_PREFIX;
extern const char* BM_WARN_PREFIX;
extern const char* BM_INFO_PREFIX;
extern const char* BM_DEBUG_PREFIX;

#define bm_error(fmt, ...) bm_printk("%s" fmt, BM_ERR_PREFIX, ## __VA_ARGS__)
#define bm_warn(fmt, ...) bm_printk("%s" fmt, BM_WARN_PREFIX, ## __VA_ARGS__)
#define bm_info(fmt, ...) bm_printk("%s" fmt, BM_INFO_PREFIX, ## __VA_ARGS__)
#define bm_debug(fmt, ...) bm_printk("%s" fmt, BM_DEBUG_PREFIX, ## __VA_ARGS__)
int bm_printk(const char* fmt, ...) __attribute__((format(printf, 1, 2)));
void bm_backtrace(void);
int bm_printk_with_backtrace(const char* fmt, ...) __attribute__((format(printf, 1, 2)));

extern void (*bm_panic)(const char* fmt, ...) __attribute__((format(printf, 1, 2)));

bool bm_ratelimit(void);

// Atomic operations
	#include <asm/atomic.h>
typedef atomic_t bm_atomic_t;

void bm_atomic_set(bm_atomic_t *atomic, int val);
int bm_atomic_read(const bm_atomic_t *atomic);
void bm_atomic_add(bm_atomic_t *atomic, int val);
void bm_atomic_sub(bm_atomic_t *atomic, int val);
int bm_atomic_add_return(bm_atomic_t *atomic, int val);
int bm_atomic_sub_return(bm_atomic_t *atomic, int val);

bool bm_compare_and_swap(volatile uint32_t* addr, uint32_t oldVal, uint32_t newVal);
bool bm_compare_and_swap64(volatile uint64_t* addr, uint64_t oldVal, uint64_t newVal);
bool bm_compare_and_swap_ptr(void* volatile* addr, void* oldVal, void* newVal);
uint32_t bm_bit_or_atomic(volatile uint32_t* addr, uint32_t mask);
uint32_t bm_bit_and_atomic(volatile uint32_t* addr, uint32_t mask);

// Threading
typedef int (*bm_thread_func_t)(void* data);

bm_task_t bm_thread_run(bm_thread_func_t func, void* data, const char* name);
void bm_thread_stop(bm_task_t thread);
bool bm_thread_should_stop(void);

bm_task_t bm_task_current(void);

int bm_task_pid(bm_task_t task);
const char* bm_task_name(bm_task_t task, char* buffer);

void bm_schedule(void);

// String functions
uint32_t bm_hash_string(const char* str, size_t bits);
int bm_strcmp(const char* str1, const char* str2);
size_t bm_strlen(const char* str);
size_t bm_strnlen(const char* str, size_t max);
char* bm_strdup(const char* str);
char* bm_strncpy(char* dst, const char* src, size_t bytes);
int bm_snprintf(char* str, size_t size, const char* fmt, ...) __attribute__((format(printf, 3, 4)));

void* bm_memcpy(void* dst, const void* src, size_t bytes);
int bm_memcmp(const void* buf1, const void* buf2, size_t bytes);
void* bm_memset(void* buf, int val, size_t bytes);
void* bm_memmove(void* dst, const void* src, size_t bytes);

// Sleeping
void bm_msleep(unsigned msec);
void bm_udelay(unsigned usec);

// Timer
struct timer_list;
typedef struct timer_list bm_timer_t;
typedef void (*bm_timer_callback_t)(void*);

uint64_t bm_absolute_time_us(void);

bm_timer_t* bm_timer_alloc(bm_timer_callback_t callback, void* data);
void bm_timer_free(bm_timer_t* timer);

void bm_timer_expire_in(bm_timer_t* timer, uint64_t ns);
void bm_timer_expire_at(bm_timer_t* timer, uint64_t ns);
void bm_timer_cancel(bm_timer_t* timer);

// Event wating
#define THREAD_UNINT         0
#define THREAD_INTERRUPTIBLE 1
#define THREAD_ABORTSAFE     2

#define THREAD_AWAKENED      0
#define THREAD_INTERRUPTED   2
#define THREAD_TIMED_OUT     3

struct bm_event;
typedef struct bm_event bm_event_t;

void bm_event_wait_prepare(bm_event_t** ev, void* event);
int bm_event_wait(bm_event_t* ev, int state);
int bm_event_wait_thread(bm_event_t* ev, int state, bm_task_t task);
int bm_event_wait_timeout(bm_event_t* ev, int state, long timeout);
void bm_event_wait_finish(bm_event_t* ev);

void bm_event_wakeup(void* event, bool one);

// User IO
#define BM_VERIFY_READ 0
#define BM_VERIFY_WRITE 1

unsigned long bm_access_ok(unsigned int type, void* addr, size_t size);

unsigned long bm_copy_to_user(void *to, const void *from, unsigned len);
unsigned long bm_copy_from_user(void *to, const void *from, unsigned len);

// Register IO
unsigned int bm_ioread8(volatile void*);
unsigned int bm_ioread16(volatile void*);
unsigned int bm_ioread16be(volatile void*);
unsigned int bm_ioread32(volatile void*);
unsigned int bm_ioread32be(volatile void*);

void bm_iowrite8(uint8_t, volatile void*);
void bm_iowrite16(uint16_t, volatile void*);
void bm_iowrite16be(uint16_t, volatile void*);
void bm_iowrite32(uint32_t, volatile void*);
void bm_iowrite32be(uint32_t, volatile void*);

// Random
uint32_t bm_random32(void);

// Min/Max
#define BM_MAX(a, b) ((a) < (b) ? (b) : (a))
#define BM_MIN(a, b) ((a) < (b) ? (a) : (b))

// Time
uint64_t bm_uptime(void);
uint64_t bm_jiffies_to_nsec(long time);
long bm_nsec_to_jiffies(uint64_t time);
void bm_realtime(uint64_t* secs, uint32_t* nanos);


#endif
