/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/delay.h>
#include <linux/hash.h>
#include <linux/interrupt.h>
#include <linux/jhash.h>
#include <linux/jiffies.h>
#include <linux/kernel.h>
#include <linux/kthread.h>
#include <linux/random.h>
#include <linux/signal.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/wait.h>
#include <asm/io.h>
#include "bm_version.h"
#if KERNEL_VERSION_OR_LATER(2, 6, 20)
	#include <linux/freezer.h>
#endif
#if KERNEL_VERSION_OR_LATER(3, 17, 0)
	#include <linux/timekeeping.h>
#else
	#include <linux/ktime.h>
#endif
#if KERNEL_VERSION_OR_LATER(4, 11, 0)
	#include <linux/sched/signal.h>
#endif
#include <asm-generic/bug.h>
#include "bm_util.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 9, 0)
	#define hlist_entry_safe(ptr, type, member) \
		({ typeof(ptr) ____ptr = (ptr); \
		   ____ptr ? hlist_entry(____ptr, type, member) : NULL; \
		})

	#undef hlist_for_each_entry
	#define hlist_for_each_entry(pos, head, member) \
		for (pos = hlist_entry_safe((head)->first, typeof(*(pos)), member); \
		     pos; \
		     pos = hlist_entry_safe((pos)->member.next, typeof(*(pos)), member))
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 25)
	#define TASK_NORMAL (TASK_INTERRUPTIBLE | TASK_UNINTERRUPTIBLE)
	#define TASK_KILLABLE TASK_INTERRUPTIBLE
#endif

// If the maximum kmalloc size is not defined, define a safe one
#ifndef KMALLOC_MAX_SIZE
	#define KMALLOC_MAX_SIZE 131072
#endif

// Basic memory management
void *bm_kmalloc(size_t size)
{
	void* ptr;
	ptr = kmalloc(size, in_interrupt() ? GFP_ATOMIC : GFP_KERNEL);
	return ptr;
}

void *bm_kzalloc(size_t size)
{
	void* ptr;
	ptr = kzalloc(size, in_interrupt() ? GFP_ATOMIC : GFP_KERNEL);
	return ptr;
}

void bm_kfree(void* mem)
{
	if (mem)
		kfree(mem);
}

void* bm_vmalloc(size_t size)
{
	void* ptr = vmalloc(size);
	return ptr;
}

void bm_vfree(void* mem)
{
	if (mem)
		vfree(mem);
}

struct bm_alloc_header
{
	uint32_t vmalloc:1,
		  offset:16;
	uint32_t reserved;
};

#define MEM_ALIGN(mem, align) (((vm_address_t)(mem) + ((align) - 1)) & ~((align) - 1))
#define MAX(a, b) ((a) > (b) ? (a) : (b))

void* bm_alloc_aligned(size_t size, uint32_t flags)
{
	bool used_vmalloc = false;
	vm_address_t mem = 0;
	vm_address_t mem_aligned;
	struct bm_alloc_header* header;
	unsigned long align = flags & BM_ALLOC_ALIGNMENT_MASK;
	size_t alloc_size = size + sizeof(struct bm_alloc_header);

	if (align > 1)
	{
		align = align > PAGE_SIZE ? PAGE_SIZE : align;
		alloc_size += align - 1;
	}

	if (get_order(alloc_size) < MAX_ORDER)
		mem = (vm_address_t)kmalloc(alloc_size, GFP_KERNEL);

	if (mem == 0 && (flags & BM_ALLOC_CONTIGUOUS) == 0)
	{
		// Couldn't allocate with kmalloc, so try vmalloc
		used_vmalloc = true;
		mem = (vm_address_t)vmalloc(alloc_size);
	}

	if (!mem)
		return NULL;

	mem_aligned = mem + sizeof(struct bm_alloc_header);
	mem_aligned = MEM_ALIGN(mem_aligned, align);
	BUG_ON(mem_aligned + size > mem + alloc_size);

	header = (struct bm_alloc_header*)mem_aligned - 1;
	BUG_ON((vm_address_t)header < mem);

	header->vmalloc = used_vmalloc;
	header->offset = mem_aligned - mem;

	return (void*)mem_aligned;
}

void bm_free_aligned(void* mem)
{
	struct bm_alloc_header* header;
	void* unaligned_mem;

	if (mem == NULL)
	{
		WARN_ON_ONCE(1);
		return;
	}

	header = (struct bm_alloc_header*)mem - 1;
	unaligned_mem = (char*)mem - header->offset;

	if (header->vmalloc)
		vfree(unaligned_mem);
	else
		kfree(unaligned_mem);
}

uint32_t bm_aligned_flags(void* mem)
{
	struct bm_alloc_header* header = (struct bm_alloc_header*)mem - 1;
	return header->vmalloc ? 0 : BM_ALLOC_CONTIGUOUS;
}

// Logging
const char *BM_ERR_PREFIX =   KERN_ERR;
const char *BM_WARN_PREFIX =  KERN_WARNING;
const char *BM_INFO_PREFIX =  KERN_INFO;
const char *BM_DEBUG_PREFIX = KERN_DEBUG;

int bm_printk(const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
		r = vprintk(fmt, args);
	va_end(args);

	return r;
}

void bm_backtrace()
{
	dump_stack();
}

int bm_printk_with_backtrace(const char* fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
		 r = vprintk(fmt, args);
	va_end(args);

	dump_stack();

	return r;
}

void (*bm_panic)(const char* fmt, ...) = panic;

bool bm_ratelimit(void)
{
	return printk_ratelimit();
}

// Atomic operations
void bm_atomic_set(bm_atomic_t *atomic, int val)
{
	atomic_set(atomic, val);
}

int bm_atomic_read(const bm_atomic_t *atomic)
{
	return atomic_read(atomic);
}

void bm_atomic_add(bm_atomic_t *atomic, int val)
{
	atomic_add(val, atomic);
}

void bm_atomic_sub(bm_atomic_t *atomic, int val)
{
	atomic_sub(val, atomic);
}

int bm_atomic_add_return(bm_atomic_t *atomic, int val)
{
	return atomic_add_return(val, atomic);
}

int bm_atomic_sub_return(bm_atomic_t *atomic, int val)
{
	return atomic_sub_return(val, atomic);
}

bool bm_compare_and_swap(volatile uint32_t* addr, uint32_t oldVal, uint32_t newVal)
{
	return cmpxchg(addr, oldVal, newVal) == oldVal;
}

bool bm_compare_and_swap64(volatile uint64_t* addr, uint64_t oldVal, uint64_t newVal)
{
	return cmpxchg64(addr, oldVal, newVal) == oldVal;
}

bool bm_compare_and_swap_ptr(void* volatile* addr, void* oldVal, void* newVal)
{
	return cmpxchg(addr, oldVal, newVal) == oldVal;
}

uint32_t bm_bit_or_atomic(volatile uint32_t* addr, uint32_t mask)
{
	uint32_t old;
	do
	{
		old = *addr;
	} while (cmpxchg(addr, old, old | mask) != old);
	return old;
}

uint32_t bm_bit_and_atomic(volatile uint32_t* addr, uint32_t mask)
{
	uint32_t old;
	do
	{
		old = *addr;
	} while (cmpxchg(addr, old, old & mask) != old);
	return old;
}


// Threading
bm_task_t bm_thread_run(bm_thread_func_t func, void* data, const char* name)
{
	return kthread_run(func, data, name);
}

void bm_thread_stop(bm_task_t thread)
{
	kthread_stop(thread);
}

bool bm_thread_should_stop(void)
{
	return kthread_should_stop();
}

bm_task_t bm_task_current(void)
{
	return current;
}

void bm_schedule(void)
{
	schedule();
}

int bm_task_pid(bm_task_t task)
{
#if KERNEL_VERSION_OR_LATER(2, 6, 24)
	return task_pid_nr(task);
#else
	return task->pid;
#endif
}

const char* bm_task_name(bm_task_t task, char* buffer)
{
	task_lock(task);
	strncpy(buffer, task->comm, TASK_COMM_LEN);
	task_unlock(task);
	return buffer;
}

// String functions
uint32_t bm_hash_string(const char* str, size_t bits)
{
	return jhash(str, strlen(str), 0) >> (32 - bits);
}

int bm_strcmp(const char* str1, const char* str2)
{
	return strcmp(str1, str2);
}

size_t bm_strlen(const char* str)
{
	return strlen(str);
}

size_t bm_strnlen(const char* str, size_t max)
{
	return strnlen(str, max);
}

char* bm_strdup(const char* str)
{
	return kstrdup(str, in_interrupt() ? GFP_ATOMIC : GFP_KERNEL);
}

char* bm_strncpy(char* dst, const char* src, size_t bytes)
{
	return strncpy(dst, src, bytes);
}

int bm_snprintf(char* str, size_t size, const char* fmt, ...)
{
	int r;
	va_list args;

	va_start(args, fmt);
	r = vsnprintf(str, size, fmt, args);
	va_end(args);

	return r;
}

void* bm_memcpy(void* dst, const void* src, size_t bytes)
{
	return memcpy(dst, src, bytes);
}

void* bm_memmove(void* dst, const void* src, size_t bytes)
{
	return memmove(dst, src, bytes);
}

int bm_memcmp(const void* buf1, const void* buf2, size_t bytes)
{
	return memcmp(buf1, buf2, bytes);
}

void* bm_memset(void* buf, int val, size_t bytes)
{
	return memset(buf, val, bytes);
}

// Sleeping
void bm_msleep(unsigned msec)
{
	msleep(msec);
}

void bm_udelay(unsigned usec)
{
	udelay(usec);
}

// Timer
uint64_t bm_absolute_time_us(void)
{
	return jiffies_to_usecs(jiffies);
}

#if KERNEL_VERSION_OR_LATER(4, 15, 0)
struct bm_timer_wrapper
{
	bm_timer_t timer;
	bm_timer_callback_t callback;
	void* data;
};

static void timer_handler(struct timer_list* timer)
{
	struct bm_timer_wrapper* timer_wrapper = container_of(timer, struct bm_timer_wrapper, timer);
	timer_wrapper->callback(timer_wrapper->data);
}
#endif

bm_timer_t* bm_timer_alloc(bm_timer_callback_t callback, void* data)
{
#if KERNEL_VERSION_OR_LATER(4, 15, 0)
	struct bm_timer_wrapper* timer = bm_kmalloc(sizeof(struct bm_timer_wrapper));
	if (!timer)
		return NULL;

	timer_setup(&timer->timer, timer_handler, 0);
	timer->callback = callback;
	timer->data = data;

	return &timer->timer;
#else
	bm_timer_t* timer = bm_kmalloc(sizeof(bm_timer_t));
	if (!timer)
		return NULL;

	init_timer(timer);
	timer->function = (void(*)(unsigned long))callback;
	timer->data = (long)data;

	return timer;
#endif
}

void bm_timer_free(bm_timer_t* timer)
{
#if KERNEL_VERSION_OR_LATER(4, 15, 0)
	struct bm_timer_wrapper* timer_wrapper = container_of(timer, struct bm_timer_wrapper, timer);
	del_timer(timer);
	bm_kfree(timer_wrapper);
#else
	del_timer(timer);
	bm_kfree(timer);
#endif
}

void bm_timer_expire_in(bm_timer_t* timer, uint64_t ns)
{
	mod_timer(timer, jiffies + (ns * HZ + NSEC_PER_SEC - 1) / NSEC_PER_SEC);
}

void bm_timer_expire_at(bm_timer_t* timer, uint64_t ns)
{
	mod_timer(timer, (ns * HZ + NSEC_PER_SEC - 1) / NSEC_PER_SEC);
}

void bm_timer_cancel(bm_timer_t* timer)
{
	del_timer_sync(timer);
}

// Event waiting
#define EVENT_TABLE_BITS 6
#define EVENT_TABLE_SIZE (1 << EVENT_TABLE_BITS)

struct bm_event
{
	struct hlist_node list;
	wait_queue_head_t queue;
	int ref;
	void* event;
};

typedef struct bm_waiter
{
#if KERNEL_VERSION_OR_LATER(4, 13, 0)
	wait_queue_entry_t wait;
#else
	wait_queue_t wait;
#endif
	bool triggered;
} bm_waiter_t;

typedef struct bm_event_table
{
	spinlock_t lock;
	int count;
	struct hlist_head events[EVENT_TABLE_SIZE];
} bm_event_table_t;

static bm_event_table_t event_table;

static inline void init_event_table(void)
{
	unsigned i;

	spin_lock_init(&event_table.lock);
	event_table.count = 0;

	for (i = 0; i < EVENT_TABLE_SIZE; ++i)
		INIT_HLIST_HEAD(&event_table.events[i]);
}

bm_event_t* get_event(void* event, bool create)
{
	bm_event_t* ev;
	unsigned idx = hash_ptr(event, EVENT_TABLE_BITS);
	unsigned long flags;

	spin_lock_irqsave(&event_table.lock, flags);
		hlist_for_each_entry(ev, &event_table.events[idx], list)
		{
			if (ev->event == event)
			{
				++ev->ref;
				goto done;
			}
		}

		if (create)
		{
			ev = kmalloc(sizeof(bm_event_t), GFP_ATOMIC);
			if (!ev)
				goto done;

			*ev = (bm_event_t) {
				.list = { NULL, NULL },
				.queue = __WAIT_QUEUE_HEAD_INITIALIZER(ev->queue),
				.ref = 1,
				.event = event
			};
			hlist_add_head(&ev->list, &event_table.events[idx]);
			++event_table.count;
		}
		else
		{
			ev = NULL;
		}
done:
	spin_unlock_irqrestore(&event_table.lock, flags);
	return ev;
}

void put_event(bm_event_t* ev)
{
	unsigned long flags;
	spin_lock_irqsave(&event_table.lock, flags);
		if (--ev->ref == 0)
		{
			hlist_del(&ev->list);
			kfree(ev);
			--event_table.count;
		}
	spin_unlock_irqrestore(&event_table.lock, flags);
}

void bm_event_wait_prepare(bm_event_t** ev, void* event)
{
	*ev = get_event(event, true);
	if (!(*ev))
		return;

	spin_lock_irq(&(*ev)->queue.lock);
}

void bm_event_wait_finish(bm_event_t* ev)
{
	if (!ev)
		return;

	spin_unlock_irq(&ev->queue.lock);
	put_event(ev);
}

typedef int (*interrupted_test_func_t)(struct task_struct*);

static inline int bm_signal_pending(struct task_struct* p)
{
	if (signal_pending(p))
	{
		try_to_freeze();

		return true;
	}
	return false;
}

static inline int bm_fatal_signal_pending(struct task_struct* p)
{
	if (signal_pending(p))
	{
		try_to_freeze();

		if (unlikely(sigismember(&p->pending.signal, SIGKILL)))
			return true;
	}
	return false;
}

static inline int bm_fatal_signal_pending_thread(struct task_struct* p)
{
	return kthread_should_stop() || bm_fatal_signal_pending(p);
}

static inline int bm_signal_pending_thread(struct task_struct* p)
{
	return kthread_should_stop() || bm_signal_pending(p);
}

static inline long xlate_state(int state)
{
	switch (state)
	{
		case THREAD_INTERRUPTIBLE:
			return TASK_KILLABLE;
		case THREAD_ABORTSAFE:
			return TASK_INTERRUPTIBLE;
		default:
			return TASK_UNINTERRUPTIBLE;
	}
}

static int event_wait_common(bm_event_t* event, int state, interrupted_test_func_t func)
{
	bm_waiter_t wait;
	int r = THREAD_AWAKENED;

	init_wait(&wait.wait);
	wait.triggered = false;

	if (!event)
		return -ENOMEM;

	while (true)
	{
		wait.wait.flags &= ~WQ_FLAG_EXCLUSIVE;
#if KERNEL_VERSION_OR_LATER(4, 13, 0)
		if (list_empty(&wait.wait.entry))
#else
		if (list_empty(&wait.wait.task_list))
#endif
			__add_wait_queue(&event->queue, &wait.wait);
		set_current_state(xlate_state(state));

		if (wait.triggered)
			break;

		spin_unlock_irq(&event->queue.lock);
			if (func && func(current))
			{
				r = THREAD_INTERRUPTED;
				spin_lock_irq(&event->queue.lock);
				break;
			}

			schedule();
		spin_lock_irq(&event->queue.lock);
	}
	__remove_wait_queue(&event->queue, &wait.wait);
	__set_current_state(TASK_RUNNING);

	return r;
}

static int event_wait_timeout_common(bm_event_t* event, int state, long timeout, interrupted_test_func_t func)
{
	bm_waiter_t wait;
	int r = THREAD_AWAKENED;

	init_wait(&wait.wait);
	wait.triggered = false;

	if (!event)
		return -ENOMEM;

	while (true)
	{
		wait.wait.flags &= ~WQ_FLAG_EXCLUSIVE;
#if KERNEL_VERSION_OR_LATER(4, 13, 0)
		if (list_empty(&wait.wait.entry))
#else
		if (list_empty(&wait.wait.task_list))
#endif
			__add_wait_queue(&event->queue, &wait.wait);
		set_current_state(xlate_state(state));

		if (wait.triggered)
			break;

		spin_unlock_irq(&event->queue.lock);
			if (func && func(current))
			{
				r = THREAD_INTERRUPTED;
				spin_lock_irq(&event->queue.lock);
				break;
			}

			timeout = schedule_timeout(timeout);
		spin_lock_irq(&event->queue.lock);
		if (!timeout)
		{
			r = THREAD_TIMED_OUT;
			break;
		}
	}
	__remove_wait_queue(&event->queue, &wait.wait);
	__set_current_state(TASK_RUNNING);

	return r;
}

int bm_event_wait(bm_event_t* event, int state)
{
	interrupted_test_func_t func = NULL;

	switch (state)
	{
		case THREAD_INTERRUPTIBLE:
			func = bm_fatal_signal_pending;
			break;

		case THREAD_ABORTSAFE:
			func = bm_signal_pending;
			break;
	}

	return event_wait_common(event, state, func);
}

int bm_event_wait_thread(bm_event_t* event, int state, bm_task_t task)
{
	interrupted_test_func_t func = NULL;

	switch (state)
	{
		case THREAD_INTERRUPTIBLE:
			func = bm_fatal_signal_pending_thread;
			break;

		case THREAD_ABORTSAFE:
			func = bm_signal_pending_thread;
			break;
	}

	return event_wait_common(event, state, func);
}

int bm_event_wait_timeout(bm_event_t* event, int state, long timeout)
{
	interrupted_test_func_t func = NULL;

	switch (state)
	{
		case THREAD_INTERRUPTIBLE:
			func = bm_fatal_signal_pending;
			break;

		case THREAD_ABORTSAFE:
			func = signal_pending;
			break;
	}

	return event_wait_timeout_common(event, state, timeout, func);
}

void bm_event_wakeup(void* event, bool one)
{
	bm_event_t* ev = get_event(event, false);
	struct list_head *tmp, *next;
	int n = one ? 1 : 0;
	unsigned long flags;

	if (!ev)
		return;

	spin_lock_irqsave(&ev->queue.lock, flags);
#if KERNEL_VERSION_OR_LATER(4, 13, 0)
		list_for_each_safe(tmp, next, &ev->queue.head)
#else
		list_for_each_safe(tmp, next, &ev->queue.task_list)
#endif
		{
#if KERNEL_VERSION_OR_LATER(4, 13, 0)
			wait_queue_entry_t* curr = list_entry(tmp, wait_queue_entry_t, entry);
#else
			wait_queue_t* curr = list_entry(tmp, wait_queue_t, task_list);
#endif
			bm_waiter_t* waiter = container_of(curr, bm_waiter_t, wait);

			waiter->triggered = true;

			if (curr->func(curr, TASK_NORMAL, 0, NULL) && !--n)
				break;
		}
	spin_unlock_irqrestore(&ev->queue.lock, flags);

	put_event(ev);
}

// User IO
unsigned long bm_access_ok(unsigned int type, void* addr, size_t size)
{
#if KERNEL_VERSION_OR_LATER(5, 0, 0) || RHEL_RELEASE_OR_LATER(8, 1)
	return access_ok(addr, size);
#else
	return access_ok(type, addr, size);
#endif
}

unsigned long bm_copy_to_user(void *to, const void *from, unsigned len)
{
	return copy_to_user(to, from, len);
}

unsigned long bm_copy_from_user(void *to, const void *from, unsigned len)
{
	return copy_from_user(to, from, len);
}

// Register IO
unsigned int bm_ioread8(volatile void* addr)
{
	return ioread8((void*)addr);
}

unsigned int bm_ioread16(volatile void* addr)
{
	return ioread16((void*)addr);
}

unsigned int bm_ioread16be(volatile void* addr)
{
	return ioread16be((void*)addr);
}

unsigned int bm_ioread32(volatile void* addr)
{
	return ioread32((void*)addr);
}

unsigned int bm_ioread32be(volatile void* addr)
{
	return ioread32be((void*)addr);
}

void bm_iowrite8(uint8_t val, volatile void* addr)
{
	iowrite8(val, (void*)addr);
}

void bm_iowrite16(uint16_t val, volatile void* addr)
{
	iowrite16(val, (void*)addr);
}

void bm_iowrite16be(uint16_t val, volatile void* addr)
{
	iowrite16be(val, (void*)addr);
}

void bm_iowrite32(uint32_t val, volatile void* addr)
{
	iowrite32(val, (void*)addr);
}

void bm_iowrite32be(uint32_t val, volatile void* addr)
{
	iowrite32be(val, (void*)addr);
}

void bm_util_init()
{
	init_event_table();
}

// Random
uint32_t bm_random32(void)
{
#if KERNEL_VERSION_OR_LATER(6, 1, 0)
	return get_random_u32();
#elif KERNEL_VERSION_OR_LATER(3, 8, 0)
	return prandom_u32();
#elif KERNEL_VERSION_OR_LATER(2, 6, 19)
	return random32();
#else
	uint32_t num;
	get_random_bytes(&num, sizeof(uint32_t));
	return num;
#endif
}

uint64_t bm_uptime(void)
{
#if KERNEL_VERSION_OR_LATER(5, 6, 0)
	struct timespec64 t;
	ktime_get_raw_ts64(&t);
	return ((uint64_t)t.tv_sec * NSEC_PER_SEC) + t.tv_nsec;
#elif KERNEL_VERSION_OR_LATER(2, 6, 28)
	struct timespec t;
	getrawmonotonic(&t);
	return ((uint64_t)t.tv_sec * NSEC_PER_SEC) + t.tv_nsec;
#else
	return ((get_jiffies_64() - INITIAL_JIFFIES) * NSEC_PER_SEC) / HZ;
#endif
}

uint64_t bm_jiffies_to_nsec(long time)
{
	return ((uint64_t)time * NSEC_PER_SEC) / HZ;
}

long bm_nsec_to_jiffies(uint64_t time)
{
	return (time * HZ) / NSEC_PER_SEC;
}

void bm_realtime(uint64_t* secs, uint32_t* nanos)
{
#if KERNEL_VERSION_OR_LATER(3, 17, 0)
	struct timespec64 ts;
	ktime_get_real_ts64(&ts);
#else
	struct timespec ts; // Fails in 2038
	ktime_get_real_ts(&ts);
#endif
	*secs = ts.tv_sec;
	*nanos = ts.tv_nsec;
}
