/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/termios.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/tty_driver.h>
#include "bmio_serial.h"
#include "bmio_subdriver.h"

#ifdef i386
	#define MAX_SERIAL_LINES 32
#else
	#define MAX_SERIAL_LINES 64
#endif

BOOL_PARAM bmio_serial_enabled = true;
module_param_named(tty, bmio_serial_enabled, bool, 0644);
MODULE_PARM_DESC(tty, "Serial TTY driver is enabled");

static struct tty_driver *serial_driver = NULL;
static bmio_serial_t* serial_driver_state[MAX_SERIAL_LINES] = { NULL };

static bmio_serial_t* get_driver_by_index(int index)
{
	bmio_serial_t* serial;
	if (!serial_driver)
		return ERR_PTR(-ENODEV);

	if (index >= MAX_SERIAL_LINES)
		return ERR_PTR(-ENOENT);

	serial = serial_driver_state[index];
	if (!serial)
		return ERR_PTR(-ENODEV);

	return serial;
}

static void set_driver_by_index(int index, bmio_serial_t* serial)
{
	if (!serial_driver)
		return;

	serial_driver_state[index] = serial;
}

#if KERNEL_VERSION_OR_LATER(3, 9, 0)
	#define call_flip_function(serial, func, ...) func(&(serial)->port, ##__VA_ARGS__)
#elif KERNEL_VERSION_OR_LATER(2, 6, 33)
	#define call_flip_function(serial, func, ...) func((serial)->port.tty, ##__VA_ARGS__)
#else
	#define call_flip_function(serial, func, ...) func((serial)->tty, ##__VA_ARGS__)
#endif

static void serial_rx_handler(bmio_serial_t* serial)
{
	unsigned char byte;

	while (bmio_serial_pop_rx_byte(serial, &byte))
	{
		if (call_flip_function(serial, tty_insert_flip_char, byte, TTY_NORMAL) < 1)
		{
			printk(KERN_WARNING "BlackmagicIO: Serial read failed - No room in TTY buffer\n");
			break;
		}
	}

	call_flip_function(serial, tty_flip_buffer_push);
}

static int serial_open(struct tty_struct *tty, struct file *file)
{
	int r;

	bmio_serial_t *serial = get_driver_by_index(tty->index);
	if (IS_ERR(serial))
		return PTR_ERR(serial);

	r = bmio_serial_open(serial);
	if (r)
		return r;

	bmio_serial_port_set_rx_callback(serial, serial_rx_handler);

#if KERNEL_VERSION_OR_LATER(2, 6, 33)
	return tty_port_open(&serial->port, tty, file);
#else
	serial->tty = tty;
	return 0;
#endif
}

static void serial_close(struct tty_struct *tty, struct file *file)
{
	bmio_serial_t *serial = get_driver_by_index(tty->index);
	if (IS_ERR(serial))
		return;

	bmio_serial_port_set_rx_callback(serial, NULL);

	bmio_serial_close(serial);

#if KERNEL_VERSION_OR_LATER(2, 6, 33)
	tty_port_close(&serial->port, tty, file);
#else
	serial->tty = NULL;
#endif
}

static int serial_write(struct tty_struct *tty, const unsigned char *data, int count)
{
	bmio_serial_t *serial = get_driver_by_index(tty->index);
	if (IS_ERR(serial))
		return PTR_ERR(serial);

	return bmio_serial_write(serial, data, count);
}

#if KERNEL_VERSION_OR_LATER(5, 14, 0)
static unsigned int serial_write_room(struct tty_struct *tty)
#else
static int serial_write_room(struct tty_struct *tty)
#endif
{
	bmio_serial_t *serial = get_driver_by_index(tty->index);
	if (IS_ERR(serial))
		return PTR_ERR(serial);

	return bmio_serial_write_room(serial);
}

#if KERNEL_VERSION_OR_LATER(5, 14, 0)
static unsigned int serial_chars_in_buffer(struct tty_struct *tty)
#else
static int serial_chars_in_buffer(struct tty_struct *tty)
#endif
{
	bmio_serial_t *serial = get_driver_by_index(tty->index);
	if (IS_ERR(serial))
		return PTR_ERR(serial);

	return bmio_serial_chars_in_buffer(serial);
}

static void serial_hangup(struct tty_struct *tty)
{
	tty_port_hangup(tty->port);
}

static struct tty_operations serial_ops =
{
	.open = serial_open,
	.close = serial_close,
	.write = serial_write,
	.write_room = serial_write_room,
	.chars_in_buffer = serial_chars_in_buffer,
	.hangup = serial_hangup,
};

int bmio_serial_get_device_path(bmio_driver_t* drv, char* buffer, size_t len)
{
	size_t path_len;
	size_t i;

	if (!bmio_serial_supported(drv))
	{
		buffer[0] = '\0';
		return -ENODEV;
	}

	path_len = snprintf(buffer, len - 1, "/dev/%s%u", serial_driver->name, drv->id);

	buffer[len - 1] = '\0';

	for (i = 5; i < path_len; ++i)
		if (buffer[i] == '!')
			buffer[i] = '/';

	return 0;
}

#if KERNEL_VERSION_OR_LATER(2, 6, 33)
static struct tty_port_operations serial_port_ops =
{
	// No ops needed at this time
};
#endif

int bmio_serial_probe(bmio_driver_t* drv)
{
	void* dev = NULL;
	bmio_serial_t* serial = NULL;
	bm_pci_device_t* pci = NULL;

	if (!bmio_serial_supported(drv))
		return 0;

	if (!bmio_serial_enabled)
		return -EPERM;

	serial = kzalloc(sizeof(bmio_serial_t), GFP_KERNEL);
	if (!serial)
		return -ENOMEM;

	serial->drv = drv;

#if KERNEL_VERSION_OR_LATER(2, 6, 33)
	tty_port_init(&serial->port);
	serial->port.ops = &serial_port_ops;
#else
	serial->tty = NULL;
#endif
#if KERNEL_VERSION_OR_LATER(3, 7, 0)
	serial_driver->ports[drv->id] = &serial->port;
#endif

	set_driver_by_index(drv->id, serial);

	pci = bmio_driver_pci(drv);
	if (!pci)
	{
#if KERNEL_VERSION_OR_LATER(3, 8, 0)
		tty_port_destroy(&serial->port);
#endif
		set_driver_by_index(drv->id, NULL);
		return -ENODEV;
	}

	dev = tty_register_device(serial_driver, drv->id, &pci->pdev->dev);
	if (IS_ERR(dev))
	{
#if KERNEL_VERSION_OR_LATER(3, 8, 0)
		tty_port_destroy(&serial->port);
#endif
		set_driver_by_index(drv->id, NULL);
		return PTR_ERR(dev);
	}

	return 0;
}

void bmio_serial_remove(bmio_driver_t* drv)
{
	struct tty_struct *tty;
	bmio_serial_t* serial = get_driver_by_index(drv->id);
	if (IS_ERR(serial) || serial->drv != drv)
		return;

#if KERNEL_VERSION_OR_LATER(2, 6, 33)
	tty = tty_port_tty_get(&serial->port);
#else
	tty = serial->tty;
#endif
	if (tty)
		tty_hangup(tty);
	tty_kref_put(tty);

	tty_unregister_device(serial_driver, drv->id);
#if KERNEL_VERSION_OR_LATER(3, 8, 0)
	tty_port_destroy(&serial->port);
#endif
#if KERNEL_VERSION_OR_LATER(3, 7, 0)
	serial_driver->ports[drv->id] = NULL;
#endif
	set_driver_by_index(drv->id, NULL);

	kfree(serial);
}

struct bmio_subdriver bmio_serial_subdriver =
{
	.name = "serial",
	.probe = bmio_serial_probe,
	.remove = bmio_serial_remove
};

int bmio_serial_driver_init(void)
{
	int ret;

	struct tty_driver* serial;
#if KERNEL_VERSION_OR_LATER(5, 15, 0)
	serial = tty_alloc_driver(MAX_SERIAL_LINES, 0);
	if (IS_ERR(serial))
		return -ENOMEM;
#else

	serial = alloc_tty_driver(MAX_SERIAL_LINES);
	if (!serial)
		return -ENOMEM;
#endif

	serial->owner = THIS_MODULE;
	serial->driver_name = "blackmagic-io-serial";
	serial->name = "blackmagic!ttyio";
	serial->major = 0,
	serial->type = TTY_DRIVER_TYPE_SERIAL,
	serial->subtype = SERIAL_TYPE_NORMAL,
	serial->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV,
	serial->init_termios = tty_std_termios;
	serial->init_termios.c_cflag = B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	serial->init_termios.c_lflag = 0;
	serial->driver_state = serial_driver_state;
	tty_set_operations(serial, &serial_ops);

	ret = tty_register_driver(serial);
	if (ret)
	{
#if KERNEL_VERSION_OR_LATER(5, 15, 0)
		tty_driver_kref_put(serial);
#else
		put_tty_driver(serial);
#endif
		return ret;
	}

	serial_driver = serial;

	return bmio_driver_register_subdriver(&bmio_serial_subdriver);
}

void bmio_serial_driver_exit(void)
{
	bmio_driver_unregister_subdriver(&bmio_serial_subdriver);

	if (serial_driver)
	{
		tty_unregister_driver(serial_driver);
#if KERNEL_VERSION_OR_LATER(5, 15, 0)
		tty_driver_kref_put(serial_driver);
#else
		put_tty_driver(serial_driver);
#endif
		serial_driver = NULL;
	}
}

