/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#ifndef BMIO_AUDIO_H
#define BMIO_AUDIO_H

struct bmio_audio;
typedef struct bmio_audio bmio_audio_t;

#include "bmio_driver.h"


#include <linux/list.h>
#include <linux/kthread.h>
#include <linux/wait.h>
#include <linux/mutex.h>
#include <sound/asound.h>

typedef struct AudioOutputInterface* AudioOutputInterfacePtr;
typedef struct AudioInputInterface* AudioInputInterfacePtr;


#define BMIO_AUDIO_MAX_NUM_CHANNELS 16

typedef enum bmio_audio_stream_type
{
	BMIO_AUDIO_CAPTURE = 1 << 0,
	BMIO_AUDIO_PLAYBACK = 1 << 1,
} bmio_audio_stream_type_t;

extern const struct bmio_audio_hardware
{
	size_t buffer_size;
	uint32_t max_samples;
	uint32_t interrupt_hz;
	uint32_t max_channels;
	uint32_t sample_rate;
} bmio_audio_hardware;

struct bmio_audio
{
	bmio_driver_t* drv;
	struct list_head list;

	AudioOutputInterfacePtr output;
	AudioInputInterfacePtr input;

	struct list_head streams;

	struct snd_card* card;
	struct snd_pcm* pcm;

	struct mutex streams_lock;

	struct task_struct *thread;
	wait_queue_head_t wait;
	bool interrupt;
};


struct snd_pcm_substream;

typedef struct bmio_audio_stream
{
	unsigned int bytes_per_frame;
	unsigned int frames_per_interrupt;
	void* buf_ptr;
	snd_pcm_uframes_t buf_size;
	snd_pcm_uframes_t buf_pos;
	snd_pcm_uframes_t elapsed_pos;
	snd_pcm_uframes_t hw_ptr;
	snd_pcm_uframes_t period_size;
	bool active;
	bool owned;

	bmio_audio_t* audio;
	struct snd_pcm_substream* substream;

	struct list_head list;

	bmio_audio_stream_type_t type;

} bmio_audio_stream_t;

typedef void (*bmio_audio_interrupt_t)(bmio_audio_t*);

bool bmio_audio_device_init(bmio_driver_t* drv);
void bmio_audio_device_remove(bmio_driver_t* drv);

bool bmio_audio_init(bmio_audio_t* aud);
void bmio_audio_deinit(bmio_audio_t* aud);

void bmio_audio_set_interrupt_callback(bmio_audio_t* aud, bmio_audio_interrupt_t func);
void bmio_audio_clear_interrupt_callback(bmio_audio_t* aud);

bmio_audio_stream_t* bmio_audio_alloc_stream(bmio_audio_t* aud);
void bmio_audio_free_stream(bmio_audio_stream_t* stream);

int bmio_audio_enable(bmio_audio_stream_t* str, int channels, int sample_size);
int bmio_audio_disable(bmio_audio_stream_t* str);

snd_pcm_uframes_t bmio_audio_write_samples(bmio_audio_stream_t* str, snd_pcm_uframes_t available);
snd_pcm_uframes_t bmio_audio_read_samples(bmio_audio_stream_t* str, snd_pcm_uframes_t space);


#endif
