/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/spinlock.h>
#include "bm_version.h"
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 31)
	#include <sound/driver.h>
#endif
#include <sound/asound.h>
#include <sound/control.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/tlv.h>
#include "bmio_audio.h"
#include "bmio_subdriver.h"
#include "bm_util.h"

static inline bmio_audio_stream_t* bmio_stream(struct snd_pcm_substream* substream)
{
	return substream->runtime->private_data;
}

static LIST_HEAD(bmio_audio_devices);
static DEFINE_SPINLOCK(bmio_audio_devices_lock);

// Common

static unsigned int bmio_audio_channels[] = { 2, 8, 16 };

static struct snd_pcm_hw_constraint_list bmio_audio_channels_constraints =
{
	.count = ARRAY_SIZE(bmio_audio_channels),
	.list = bmio_audio_channels,
	.mask = 0,
};

static void bmio_audio_interrupt(bmio_audio_t* aud)
{
	aud->interrupt = true;
	wake_up_interruptible(&aud->wait);
}

static void bmio_audio_playback_update(bmio_audio_stream_t* str)
{
	struct snd_pcm_substream* substream = str->substream;
	struct snd_pcm_runtime* runtime = substream->runtime;
	snd_pcm_sframes_t avail;
	snd_pcm_uframes_t written;

	// Work out how much data is available to copy
	avail = str->hw_ptr + runtime->buffer_size - runtime->control->appl_ptr;
	if (avail < 0)
		avail += runtime->boundary;
	else if ((snd_pcm_uframes_t)avail >= runtime->boundary)
		avail -= runtime->boundary;
	avail = runtime->buffer_size - avail;

	// Copy any available data
	if (avail > 0)
	{
		written = bmio_audio_write_samples(str, avail);
		if (written > 0)
			str->hw_ptr += written;
	}

	// Move the pointer as it plays (not as it is copied)
	str->buf_pos += str->frames_per_interrupt;
	if (str->buf_pos >= str->buf_size)
		str->buf_pos %= str->buf_size;

	// Work out when the period has elapsed
	str->elapsed_pos += str->frames_per_interrupt;
	if (str->elapsed_pos >= str->period_size)
	{
		str->elapsed_pos %= str->period_size;
		snd_pcm_period_elapsed(substream);
	}
}

static void bmio_audio_capture_update(bmio_audio_stream_t* str)
{
	struct snd_pcm_substream* substream = str->substream;
	snd_pcm_uframes_t read;

	// Read any available data
	read = bmio_audio_read_samples(str, snd_pcm_capture_hw_avail(substream->runtime));
	if (read > 0)
	{
		str->buf_pos += read;
		if (str->buf_pos >= str->buf_size)
			str->buf_pos %= str->buf_size;

		// If we've copied at least a periods worth of data, signal the period over
		str->elapsed_pos += read;
		if (str->elapsed_pos >= str->period_size)
		{
			str->elapsed_pos %= str->period_size;
			snd_pcm_period_elapsed(substream);
		}
	}
}

static int bmio_audio_thread(void* data)
{
	bmio_audio_t* aud = data;
	bmio_audio_stream_t* str;

	while (true)
	{
		wait_event_interruptible(aud->wait, aud->interrupt || kthread_should_stop());
		aud->interrupt = false;

		if (kthread_should_stop())
			break;

		mutex_lock(&aud->streams_lock);

			list_for_each_entry(str, &aud->streams, list)
			{
				if (!str->active)
					continue;

				if (str->type == BMIO_AUDIO_CAPTURE)
					bmio_audio_capture_update(str);
				else if (str->type == BMIO_AUDIO_PLAYBACK)
					bmio_audio_playback_update(str);
			}

		mutex_unlock(&aud->streams_lock);
	}

	return 0;
}

static int bmio_audio_hw_params(struct snd_pcm_substream *substream, struct snd_pcm_hw_params *hw_params)
{
	return snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));
}

static int bmio_audio_hw_free(struct snd_pcm_substream *substream)
{
	return snd_pcm_lib_free_pages(substream);
}

static int bmio_audio_prepare(struct snd_pcm_substream* substream)
{
	struct snd_pcm_runtime* runtime = substream->runtime;
	bmio_audio_stream_t* str = runtime->private_data;
	int r;

	str->elapsed_pos = 0;
	str->buf_pos = 0;
	str->hw_ptr = 0;
	str->bytes_per_frame = frames_to_bytes(runtime, 1);
	str->frames_per_interrupt = runtime->rate / bmio_audio_hardware.interrupt_hz;
	str->buf_ptr = runtime->dma_area;
	str->buf_size = runtime->buffer_size;
	str->period_size = runtime->period_size;

	r = bmio_audio_enable(str, runtime->channels, snd_pcm_format_width(runtime->format));
	if (r < 0)
		return r;

	return 0;
}

static int bmio_audio_trigger(struct snd_pcm_substream* substream, int cmd)
{
	// Atomic function
	bmio_audio_stream_t* str = bmio_stream(substream);

	switch (cmd)
	{
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
		str->active = true;
		break;

	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
		str->active = false;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

static snd_pcm_uframes_t bmio_audio_pointer(struct snd_pcm_substream *substream)
{
	// Atomic function
	bmio_audio_stream_t* str = bmio_stream(substream);

	return str->buf_pos;
}

static void bmio_audio_destroy_runtime_private(struct snd_pcm_runtime* runtime)
{
	bmio_audio_free_stream(runtime->private_data);
	runtime->private_data = NULL;
}

static int bmio_audio_close(struct snd_pcm_substream* substream)
{
	bmio_audio_stream_t* str = bmio_stream(substream);

	mutex_lock(&str->audio->streams_lock);
		list_del(&str->list);
	mutex_unlock(&str->audio->streams_lock);

	return bmio_audio_disable(str);
}

static void bmio_audio_setup_hardware(struct snd_pcm_runtime* runtime)
{
	static struct snd_pcm_hardware hardware =
	{
		.info = SNDRV_PCM_INFO_MMAP |
		        SNDRV_PCM_INFO_MMAP_VALID |
		        SNDRV_PCM_INFO_INTERLEAVED |
		        SNDRV_PCM_INFO_PAUSE |
		        SNDRV_PCM_INFO_RESUME,
		.formats = SNDRV_PCM_FMTBIT_S16_LE |
		           SNDRV_PCM_FMTBIT_S32_LE,
		.channels_min = 2,
		.period_bytes_min = 64,
		.periods_min = 1,
		.periods_max = 1024,
	};

	runtime->hw = hardware;

	runtime->hw.buffer_bytes_max = bmio_audio_hardware.buffer_size;
	runtime->hw.period_bytes_max = bmio_audio_hardware.buffer_size;

	runtime->hw.rate_min = bmio_audio_hardware.sample_rate;
	runtime->hw.rate_max = bmio_audio_hardware.sample_rate;
	if (bmio_audio_hardware.sample_rate == 48000)
		runtime->hw.rates = SNDRV_PCM_RATE_48000;

	runtime->hw.channels_max = bmio_audio_hardware.max_channels;
}

// Playback

static int bmio_audio_playback_open(struct snd_pcm_substream* substream)
{
	bmio_audio_t *aud = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime* runtime = substream->runtime;
	bmio_audio_stream_t* str;
	int r;

	bmio_audio_setup_hardware(runtime);

	r = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS, &bmio_audio_channels_constraints);
	if (r)
		return r;

	str = bmio_audio_alloc_stream(aud);
	if (!str)
		return -ENOMEM;

	INIT_LIST_HEAD(&str->list);
	str->substream = substream;
	str->type = BMIO_AUDIO_PLAYBACK;

	runtime->private_data = str;
	runtime->private_free = bmio_audio_destroy_runtime_private;

	mutex_lock(&aud->streams_lock);
		list_add(&str->list, &aud->streams);
	mutex_unlock(&aud->streams_lock);

	return 0;
}

static struct snd_pcm_ops bmio_audio_playback_ops =
{
	.open = bmio_audio_playback_open,
	.close = bmio_audio_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = bmio_audio_hw_params,
	.hw_free = bmio_audio_hw_free,
	.prepare = bmio_audio_prepare,
	.trigger = bmio_audio_trigger,
	.pointer = bmio_audio_pointer,
};

// Capture

static int bmio_audio_capture_open(struct snd_pcm_substream* substream)
{
	bmio_audio_t *aud = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime* runtime = substream->runtime;
	bmio_audio_stream_t* str;

	bmio_audio_setup_hardware(runtime);

	str = bmio_audio_alloc_stream(aud);
	if (!str)
		return -ENOMEM;

	INIT_LIST_HEAD(&str->list);
	str->substream = substream;
	str->type = BMIO_AUDIO_CAPTURE;

	runtime->private_data = str;
	runtime->private_free = bmio_audio_destroy_runtime_private;

	mutex_lock(&aud->streams_lock);
		list_add(&str->list, &aud->streams);
	mutex_unlock(&aud->streams_lock);

	return 0;
}

static struct snd_pcm_ops bmio_audio_capture_ops =
{
	.open = bmio_audio_capture_open,
	.close = bmio_audio_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = bmio_audio_hw_params,
	.hw_free = bmio_audio_hw_free,
	.prepare = bmio_audio_prepare,
	.trigger = bmio_audio_trigger,
	.pointer = bmio_audio_pointer,
};

// Init/Free

#if KERNEL_VERSION_OR_LATER(3, 15, 0) || RHEL_RELEASE_OR_LATER(7, 6)
static int create_sound_card(struct device *parent, int idx, const char *xid, struct module *module, int extra_size, struct snd_card **card_ret)
{
	return snd_card_new(parent, idx, xid, module, extra_size, card_ret);
}
#elif KERNEL_VERSION_OR_LATER(2, 6, 30)
static int create_sound_card(struct device *parent, int idx, const char *xid, struct module *module, int extra_size, struct snd_card **card_ret)
{
	int r = snd_card_create(idx, xid, module, extra_size, card_ret);
	if (r == 0)
		snd_card_set_dev(*card_ret, parent);
	return r;
}
#else
static int create_sound_card(struct device *parent, int idx, const char *xid, struct module *module, int extra_size, struct snd_card **card_ret)
{
	struct snd_card* card = snd_card_new(idx, xid, module, extra_size);
	if (card == NULL)
		return -ENOMEM;

	*card_ret = card;

	return 0;
}
#endif

static int bmio_audio_init_sound_card(bmio_audio_t* aud)
{
	struct snd_card* card;
	int r;

	r = create_sound_card(bmio_driver_device(aud->drv), -1, "BlackmagicIO", THIS_MODULE, 0, &card);
	if (r < 0)
		return r;

	aud->card = card;
	card->private_data = aud;

	snprintf(card->longname, 80, "BlackmagicIO Audio Device (%s)", bmio_driver_name(aud->drv));
	snprintf(card->shortname, 32, "BlackmagicIO Audio");
	strcpy(card->driver, "bmdio-audio");

	return 0;
}

static int bmio_audio_init_pcm(bmio_audio_t* aud)
{
	static const char* PCM_DEVICE_NAME = "PCM";
	int err;
	struct snd_pcm* pcm;
	uint32_t type = BMIO_AUDIO_CAPTURE | BMIO_AUDIO_PLAYBACK;
	int playback_substreams = 1;
	int capture_substreams = 1;

	if (!(type & BMIO_AUDIO_PLAYBACK))
		playback_substreams = 0;
	if (!(type & BMIO_AUDIO_CAPTURE))
		capture_substreams = 0;

	err = snd_pcm_new(aud->card, (char*)PCM_DEVICE_NAME, 0, playback_substreams, capture_substreams, &pcm);
	if (err)
		return err;

	aud->pcm = pcm;
	pcm->private_data = aud;

	pcm->info_flags = 0;
	strcpy(pcm->name, PCM_DEVICE_NAME);

	if (type & BMIO_AUDIO_PLAYBACK)
		snd_pcm_set_ops(aud->pcm, SNDRV_PCM_STREAM_PLAYBACK, &bmio_audio_playback_ops);
	if (type & BMIO_AUDIO_CAPTURE)
		snd_pcm_set_ops(aud->pcm, SNDRV_PCM_STREAM_CAPTURE, &bmio_audio_capture_ops);

	snd_pcm_lib_preallocate_pages_for_all(pcm,
		SNDRV_DMA_TYPE_CONTINUOUS,
		snd_dma_continuous_data(GFP_KERNEL),
		bmio_audio_hardware.buffer_size,
		bmio_audio_hardware.buffer_size);

	return 0;
}

static void bmio_audio_free(bmio_audio_t* aud)
{
	if (!aud)
		return;

	spin_lock(&bmio_audio_devices_lock);
		list_del(&aud->list);
	spin_unlock(&bmio_audio_devices_lock);

	bmio_audio_clear_interrupt_callback(aud);

	if (aud->thread)
		kthread_stop(aud->thread);

	if (aud->card)
		snd_card_free(aud->card);
	kfree(aud);
}

static int bmio_audio_probe(bmio_driver_t* drv)
{
	bmio_audio_t* aud;
	int err = 0;

	aud = kzalloc(sizeof(bmio_audio_t), GFP_KERNEL);
	if (!aud)
		return -ENOMEM;

	aud->drv = drv;
	aud->thread = NULL;
	aud->interrupt = false;
	INIT_LIST_HEAD(&aud->streams);
	INIT_LIST_HEAD(&aud->list);
	mutex_init(&aud->streams_lock);
	init_waitqueue_head(&aud->wait);

	err = bmio_audio_init_sound_card(aud);
	if (err < 0)
		goto bail;

	err = bmio_audio_init_pcm(aud);
	if (err < 0)
		goto bail;

	if (!bmio_audio_init(aud))
	{
		err = -EINVAL;
		goto bail;
	}

	aud->thread = kthread_run(bmio_audio_thread, aud, "blackmagic-io-audio:%u", bmio_driver_id(drv));
	if (!aud->thread)
	{
		err = -ESRCH;
		goto bail;
	}

	bmio_audio_set_interrupt_callback(aud, bmio_audio_interrupt);

	err = snd_card_register(aud->card);
	if (err < 0)
		goto bail;

	spin_lock(&bmio_audio_devices_lock);
		list_add(&aud->list, &bmio_audio_devices);
	spin_unlock(&bmio_audio_devices_lock);

	return 0;

bail:
	bmio_audio_free(aud);
	return err;
}

static void bmio_audio_remove(bmio_driver_t* drv)
{
	bmio_audio_t* aud;

	spin_lock(&bmio_audio_devices_lock);

	list_for_each_entry(aud, &bmio_audio_devices, list)
	{
		if (aud->drv == drv)
		{
			spin_unlock(&bmio_audio_devices_lock);
			bmio_audio_free(aud);
			return;
		}
	}

	spin_unlock(&bmio_audio_devices_lock);
}

bmio_subdriver_t bmio_audio_subdriver =
{
	.name = "ALSA",
	.probe = bmio_audio_probe,
	.remove = bmio_audio_remove
};

static int __init bmio_init(void)
{
	return bmio_driver_register_subdriver(&bmio_audio_subdriver);
}

static void __exit bmio_exit(void)
{
	bmio_driver_unregister_subdriver(&bmio_audio_subdriver);
}

MODULE_AUTHOR("Blackmagic Design Inc. <developer@blackmagicdesign.com>");
MODULE_DESCRIPTION("Blackmagic Design BlackmagicIO ALSA driver");
MODULE_VERSION("12.4.1a15");
MODULE_LICENSE("Proprietary");

module_init(bmio_init);
module_exit(bmio_exit);
