/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/moduleparam.h>
#include "bm_pci.h"
#include "bm_util.h"
#include "bm_version.h"

char* default_irq_type = "msi";
module_param_named(irq, default_irq_type, charp, 0444);
MODULE_PARM_DESC(irq, "The default irq type: msi, legacy");

bm_pci_device_t* bm_pci_alloc(struct pci_dev* pdev)
{
	bm_pci_device_t* pci = kzalloc(sizeof(bm_pci_device_t), GFP_KERNEL);
	if (!pci)
		return NULL;

	pci->pdev = pci_dev_get(pdev);
	atomic_set(&pci->ref, 1);
	return pci;
}

bm_pci_device_t* bm_pci_get(bm_pci_device_t* pci)
{
	atomic_inc(&pci->ref);
	return pci;
}

void bm_pci_put(bm_pci_device_t* pci)
{
	if (atomic_sub_and_test(1, &pci->ref))
	{
		pci_dev_put(pci->pdev);
		kfree(pci);
	}
}

bool bm_pci_start(bm_pci_device_t* pci)
{
	if (pci_enable_device(pci->pdev) < 0)
		return false;

	pci_set_master(pci->pdev);

	if (dma_set_mask(&(pci->pdev)->dev, DMA_BIT_MASK(64)) < 0)
	{
		if (dma_set_mask(&(pci->pdev)->dev, DMA_BIT_MASK(32)) < 0)
			goto bail;
	}

	if (strcmp(default_irq_type, "msi") == 0)
		pci_enable_msi(pci->pdev);

	return true;

bail:
	pci_disable_device(pci->pdev);
	return false;
}

void bm_pci_stop(bm_pci_device_t* pci)
{
	if (pci->pdev->msi_enabled)
		pci_disable_msi(pci->pdev);
	pci_disable_device(pci->pdev);
}

void pci_clear_master(struct pci_dev *dev)
{
	uint16_t old_cfg;
	uint16_t cfg;

	pci_read_config_word(dev, PCI_COMMAND, &old_cfg);
	cfg = old_cfg & ~PCI_COMMAND_MASTER;
	if (cfg != old_cfg)
		pci_write_config_word(dev, PCI_COMMAND, cfg);
	dev->is_busmaster = false;
}

bool bm_pci_set_bus_master_enable(bm_pci_device_t* pci, bool enable)
{
	bool ret = pci->pdev->is_busmaster;
	if (enable)
		pci_set_master(pci->pdev);
	else
		pci_clear_master(pci->pdev);
	return ret;
}

bool bm_pci_set_memory_enable(bm_pci_device_t* pci, bool enable)
{
	uint16_t cfg;
	uint16_t old_cfg;

	pci_read_config_word(pci->pdev, PCI_COMMAND, &old_cfg);

	if (enable)
		cfg = old_cfg | PCI_COMMAND_MEMORY;
	else
		cfg = old_cfg & ~PCI_COMMAND_MEMORY;

	if (cfg != old_cfg)
		pci_write_config_word(pci->pdev, PCI_COMMAND, cfg);

	return old_cfg & PCI_COMMAND_MEMORY;
}

uint8_t bm_pci_bus_number(bm_pci_device_t* pci)
{
	return pci->pdev->bus->number;
}

uint8_t bm_pci_device_number(bm_pci_device_t* pci)
{
	return PCI_SLOT(pci->pdev->devfn);
}

uint8_t bm_pci_function_number(bm_pci_device_t* pci)
{
	return PCI_FUNC(pci->pdev->devfn);
}

uint8_t bm_pci_read_config_u8(bm_pci_device_t* pci, int offset)
{
	uint8_t val;
	pci_read_config_byte(pci->pdev, offset, &val);
	return val;
}

uint16_t bm_pci_read_config_u16(bm_pci_device_t* pci, int offset)
{
	uint16_t val;
	pci_read_config_word(pci->pdev, offset, &val);
	return val;
}

uint32_t bm_pci_read_config_u32(bm_pci_device_t* pci, int offset)
{
	uint32_t val;
	pci_read_config_dword(pci->pdev, offset, &val);
	return val;
}

void bm_pci_write_config_u8(bm_pci_device_t* pci, int offset, uint8_t val)
{
	pci_write_config_byte(pci->pdev, offset, val);
}

void bm_pci_write_config_u16(bm_pci_device_t* pci, int offset, uint16_t val)
{
	pci_write_config_word(pci->pdev, offset, val);
}

void bm_pci_write_config_u32(bm_pci_device_t* pci, int offset, uint32_t val)
{
	pci_write_config_dword(pci->pdev, offset, val);
}

bool bm_pci_using_msi(bm_pci_device_t* pci)
{
	return pci->pdev->msi_enabled;
}

#if KERNEL_VERSION_OR_LATER(2, 6, 19)
	#define bm_interrupt_handler(irq, data, regs) bm_interrupt_handler(irq, data)
#endif

static irqreturn_t bm_interrupt_handler(int irq, void* data, struct pt_regs* regs)
{
	bm_pci_device_t* pci = (bm_pci_device_t*)data;
	return pci->irq_handler(irq, pci->irq_data);
}

bool bm_pci_register_interrupt(bm_pci_device_t* pci, const char* name, bm_interrupt_handler_t handler, void* data)
{
	unsigned long flags = 0;

	if (!pci->pdev->msi_enabled)
		flags |= IRQF_SHARED;

	if (pci->irq_data)
		return false;

	pci->irq_name[BM_PCI_IRQ_NAME_SIZE - 1] = '\0';
	strncpy(pci->irq_name, name, BM_PCI_IRQ_NAME_SIZE - 1);

	pci->irq_data = data;
	pci->irq_handler = handler;


	if (request_irq(pci->pdev->irq, bm_interrupt_handler, flags, pci->irq_name, pci) < 0)
		return false;

	return true;
}

void bm_pci_unregister_interrupt(bm_pci_device_t* pci)
{
	unsigned int irq = pci->pdev->irq;

	if (!pci->irq_data)
		return;


	free_irq(irq, pci);
	pci->irq_data = NULL;
}

bool bm_pci_map_bar(bm_pci_device_t* pci, int bar, void** addr, vm_size_t* size)
{
	*addr = pcim_iomap(pci->pdev, bar, 0 /* not set */);
	if (!*addr)
	{
		void __iomem **tbl = (void __iomem **)pcim_iomap_table(pci->pdev);
		if (tbl)
			*addr = tbl[bar];
	}

	if (!*addr)
		return false;

	*size = pci_resource_len(pci->pdev, bar);
	return true;
}
