/* -LICENSE-START-
** Copyright (c) 2013 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/
#include <linux/list.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include "bm_version.h"
#if KERNEL_VERSION_OR_LATER(4, 11, 0)
	#include <linux/sched/debug.h>
#endif
#include "bm_locks.h"
#include "bm_util.h"

/* Spin locks */
bm_spinlock_t* bm_spin_alloc(void)
{
	bm_spinlock_t* lock = bm_kmalloc(sizeof(bm_spinlock_t));
	if (!lock)
		return NULL;

	spin_lock_init(lock);
	return lock;
}

void bm_spin_free(bm_spinlock_t* lock)
{
	bm_kfree(lock);
}

void bm_spin_lock(bm_spinlock_t* lock)
{
	spin_lock(lock);
}

bool bm_spin_trylock(bm_spinlock_t* lock)
{
	return (spin_trylock(lock) != 0);
}

void bm_spin_unlock(bm_spinlock_t* lock)
{
	spin_unlock(lock);
}

void bm_spin_lock_irq(bm_spinlock_t* lock)
{
	spin_lock_irq(lock);
}

void bm_spin_unlock_irq(bm_spinlock_t* lock)
{
	spin_unlock_irq(lock);
}

void bm_spin_lock_irqsave(bm_spinlock_t* lock, unsigned long* flags)
{
	spin_lock_irqsave(lock, *flags);
}

void bm_spin_unlock_irqrestore(bm_spinlock_t* lock, unsigned long flags)
{
	spin_unlock_irqrestore(lock, flags);
}

/* Read/Write spin locks */
bm_rwlock_t* bm_rwlock_alloc(void)
{
	bm_rwlock_t* lock = bm_kmalloc(sizeof(bm_rwlock_t));
	if (!lock)
		return NULL;

	rwlock_init(lock);
	return lock;
}

void bm_rwlock_free(bm_rwlock_t* lock)
{
	bm_kfree(lock);
}

void bm_rwlock_read_lock(bm_rwlock_t* lock)
{
	read_lock(lock);
}

bool bm_rwlock_read_trylock(bm_rwlock_t* lock)
{
	return (read_trylock(lock) != 0);
}

void bm_rwlock_read_unlock(bm_rwlock_t* lock)
{
	read_unlock(lock);
}

void bm_rwlock_read_lock_irq(bm_rwlock_t* lock)
{
	read_lock_irq(lock);
}

void bm_rwlock_read_unlock_irq(bm_rwlock_t* lock)
{
	read_unlock_irq(lock);
}

void bm_rwlock_read_lock_irqsave(bm_rwlock_t* lock, unsigned long* flags)
{
	read_lock_irqsave(lock, *flags);
}

void bm_rwlock_read_unlock_irqrestore(bm_rwlock_t* lock, unsigned long flags)
{
	read_unlock_irqrestore(lock, flags);
}

void bm_rwlock_write_lock(bm_rwlock_t* lock)
{
	write_lock(lock);
}

bool bm_rwlock_write_trylock(bm_rwlock_t* lock)
{
	return (write_trylock(lock) != 0);
}

void bm_rwlock_write_unlock(bm_rwlock_t* lock)
{
	write_unlock(lock);
}

void bm_rwlock_write_lock_irq(bm_rwlock_t* lock)
{
	write_lock_irq(lock);
}

void bm_rwlock_write_unlock_irq(bm_rwlock_t* lock)
{
	write_unlock_irq(lock);
}

void bm_rwlock_write_lock_irqsave(bm_rwlock_t* lock, unsigned long* flags)
{
	write_lock_irqsave(lock, *flags);
}

void bm_rwlock_write_unlock_irqrestore(bm_rwlock_t* lock, unsigned long flags)
{
	write_unlock_irqrestore(lock, flags);
}

/* Basic mutex */
bm_mutex_t* bm_mutex_alloc()
{
	bm_mutex_t* mutex = bm_kmalloc(sizeof(bm_mutex_t));
	mutex_init(mutex);
	return mutex;
}

void bm_mutex_free(bm_mutex_t* mutex)
{
	bm_kfree(mutex);
}

void bm_mutex_lock(bm_mutex_t* mutex)
{
	mutex_lock(mutex);
}

bool bm_mutex_trylock(bm_mutex_t* mutex)
{
	return (mutex_trylock(mutex) != 0);
}

void bm_mutex_unlock(bm_mutex_t* mutex)
{
	mutex_unlock(mutex);
}

int bm_mutex_sleep(bm_mutex_t* mutex, void* event, int state)
{
	int r = 0;
	bm_event_t* ev;

	bm_event_wait_prepare(&ev, event);
	bm_mutex_unlock(mutex);
	r = bm_event_wait(ev, state);
	bm_event_wait_finish(ev);
	bm_mutex_lock(mutex);

	return r;
}

int bm_mutex_sleep_timeout(bm_mutex_t* mutex, void* event, int state, long timeout)
{
	int r = 0;
	bm_event_t* ev;

	bm_event_wait_prepare(&ev, event);
	bm_mutex_unlock(mutex);
	r = bm_event_wait_timeout(ev, state, timeout);
	bm_event_wait_finish(ev);
	bm_mutex_lock(mutex);

	return r;
}

void bm_mutex_wakeup(bm_mutex_t* mutex, void* event, bool one)
{
	bm_event_wakeup(event, one);
}

/* Recursive mutex */
#if KERNEL_VERSION_OR_LATER(2, 6, 33)
	#define RAW_SPIN_LOCK_UNLOCKED(...) __RAW_SPIN_LOCK_UNLOCKED(__VA_ARGS__)
#else
	// Newer version spinlocks can sleep. raw_spinlocks do not
	#define raw_spinlock_t spinlock_t
	#define raw_spin_lock_irqsave spin_lock_irqsave
	#define raw_spin_unlock_irqrestore spin_unlock_irqrestore
	#define raw_spin_lock_irq spin_lock_irq
	#define raw_spin_unlock_irq spin_unlock_irq
	#define RAW_SPIN_LOCK_UNLOCKED(...) SPIN_LOCK_UNLOCKED
#endif

struct bm_rmutex_waiter
{
	struct task_struct* task;
	struct list_head list;
};

struct bm_rmutex
{
	raw_spinlock_t lock;
	int32_t count;
	struct list_head wait_list;
	struct task_struct* task;
};

struct bm_rmutex* bm_rmutex_alloc(void)
{
	struct bm_rmutex* mutex = bm_kmalloc(sizeof(struct bm_rmutex));
	*mutex = (struct bm_rmutex) {
		.lock = RAW_SPIN_LOCK_UNLOCKED(mutex->lock),
		.count = 0,
		.wait_list = LIST_HEAD_INIT(mutex->wait_list),
		.task = NULL
	};
	return mutex;
}

void bm_rmutex_free(struct bm_rmutex* mutex)
{
	bm_kfree(mutex);
}

void __sched bm_rmutex_lock(struct bm_rmutex* mutex)
{
	long timeout = MAX_SCHEDULE_TIMEOUT;
	unsigned long flags;

	raw_spin_lock_irqsave(&mutex->lock, flags);

	if (likely(mutex->count == 0 || mutex->task == current))
	{
		mutex->task = current;
		++mutex->count;
	}
	else
	{
		struct bm_rmutex_waiter waiter;
		list_add_tail(&waiter.list, &mutex->wait_list);
		waiter.task = current;

		for (;;)
		{
			if (timeout <= 0)
			{
				list_del(&waiter.list);
				break;
			}

			__set_current_state(TASK_UNINTERRUPTIBLE);
			raw_spin_unlock_irq(&mutex->lock);
			timeout = schedule_timeout(timeout);
			raw_spin_lock_irq(&mutex->lock);

			if (likely(mutex->task == current))
				break;
		}
	}
	raw_spin_unlock_irqrestore(&mutex->lock, flags);
}

bool bm_rmutex_trylock(struct bm_rmutex* mutex)
{
	unsigned long flags;
	bool success = false;
	struct task_struct *task;

	raw_spin_lock_irqsave(&mutex->lock, flags);

	task = current;
	if (likely(mutex->count == 0 || mutex->task == task))
	{
		mutex->task = task;
		++mutex->count;
		success = true;
	}

	raw_spin_unlock_irqrestore(&mutex->lock, flags);
	return success;
}

void bm_rmutex_unlock(struct bm_rmutex* mutex)
{
	unsigned long flags;

	raw_spin_lock_irqsave(&mutex->lock, flags);

	--mutex->count;
	if (likely(mutex->count == 0))
	{
		mutex->task = NULL;
		if (!list_empty(&mutex->wait_list))
		{
			struct bm_rmutex_waiter* waiter = list_first_entry(&mutex->wait_list,
				struct bm_rmutex_waiter, list);
			list_del(&waiter->list);
			mutex->task = waiter->task;
			++mutex->count;
			wake_up_process(waiter->task);
		}
	}
	raw_spin_unlock_irqrestore(&mutex->lock, flags);
}

bool bm_rmutex_have_lock(const struct bm_rmutex* mutex)
{
	unsigned long flags;
	bool have_lock;

	raw_spin_lock_irqsave((raw_spinlock_t*)&mutex->lock, flags);
	have_lock = (mutex->task == current);
	raw_spin_unlock_irqrestore((raw_spinlock_t*)&mutex->lock, flags);

	return have_lock;
}

